#include <vector>
#include <variant>
#include <iostream>
#include <chrono>
#include <cstdint>
#include <fstream>

#include <opencv2/opencv.hpp>
#include <opencv2/core.hpp>
#include <opencv2/calib3d.hpp>

#include <nlohmann/json.hpp>
using json = nlohmann::json;

#include "feature_based_vo.hpp"
#include "detector_wrappers.hpp"
#include "detector_configurations.hpp"

Feature_Based_Visual_Odometry::Feature_Based_Visual_Odometry() {
    // Debugging 
    debug = true;

    // Camera and frame configuration
    undistort_frame = true;
    greyscale = true;
    scale_factor = 1;
    frame_size = cv::Size(1280, 720); // Default for KuupKulgur camera
    camera_matrix = (cv::Mat_<double>(3, 3) << 
                    811.27847985, 0, 511.5733993,
                    0, 806.82258429, 295.02167692,
                    0, 0, 1);
    dist_coeffs = (cv::Mat_<double>(1, 5) << -0.43500715, 0.28144541, 0.00110741, -0.00460803, -0.13176509);

    resized_frame_size = cv::Size(frame_size.width * scale_factor, frame_size.height * scale_factor);
    aspect_minus_one = 0;
    camera_matrix.at<double>(0,0) *= scale_factor; // fx, fl
    camera_matrix.at<double>(1,1) *= scale_factor; // fy, fl
    camera_matrix.at<double>(0,2) *= scale_factor; // cx, ppx
    camera_matrix.at<double>(1,2) *= scale_factor / (1 + aspect_minus_one); // cy, ppy
    
    new_camera_matrix = cv::getOptimalNewCameraMatrix(camera_matrix, dist_coeffs, resized_frame_size, 1, resized_frame_size, &frame_roi);

    // Feature detection and matching setup
    detector_config = detector_configuration{SIFT_CPU_config{}};
    detector = i_detector::create(detector_type::SIFT_CPU ,detector_configuration{SIFT_CPU_config{}});

    // RANSAC configuration
    RANSAC_tries = 15;
    RANSAC_iterations = 100;
    RANSAC_reproj_error = 8.0f;
    RANSAC_reproj_error_spread = 2.0f;
}

Feature_Based_Visual_Odometry::Feature_Based_Visual_Odometry(json json_config){
    // Debugging 
    debug = json_config["debug"]["enabled"];
    frame_time_ms = json_config["debug"]["frame_time"];

    // Camera and frame configuration
    undistort_frame = json_config["camera"]["undistort"];
    greyscale = json_config["camera"]["greyscale"];
    scale_factor = json_config["camera"]["scale_factor"];
    frame_size = cv::Size(json_config["camera"]["frame_size"]["width"], json_config["camera"]["frame_size"]["height"]);

    for (int i = 0; i < 3; ++i) {
        for (int j = 0; j < 3; ++j) {
            camera_matrix.at<double>(i, j) = json_config["camera"]["camera_matrix"][i][j];
        }
    }

    for (int i = 0; i < 5; ++i) {
        dist_coeffs.at<double>(0, i) = json_config["camera"]["distortion_coefficients"][i];
    }

    // Calculate new downscaled frame size, focal length?, camera matri
    resized_frame_size = cv::Size(frame_size.width * scale_factor, frame_size.height * scale_factor);
    // new_camera_matrix = cv::getOptimalNewCameraMatrix(camera_matrix, dist_coeffs, frame_size, 1, frame_size, &frame_roi);

    aspect_minus_one = json_config["camera"]["aspect_minus_one"];
    camera_matrix.at<double>(0,0) *= scale_factor; // fx, fl
    camera_matrix.at<double>(1,1) *= scale_factor; // fy, fl
    camera_matrix.at<double>(0,2) *= scale_factor; // cx, ppx
    camera_matrix.at<double>(1,2) *= scale_factor / (1 + aspect_minus_one); // cy, ppy

    new_camera_matrix = cv::getOptimalNewCameraMatrix(camera_matrix, dist_coeffs, resized_frame_size, 1, resized_frame_size, &frame_roi);

    // Feature detector configuration
    detector = i_detector::create(json_config["feature_detector"]);

    // RANSAC configuration
    RANSAC_tries = json_config["RANSAC"]["RANSAC_tries"];
    RANSAC_iterations = json_config["RANSAC"]["RANSAC_iterations"];
    RANSAC_reproj_error = json_config["RANSAC"]["RANSAC_reproj_error"];
    RANSAC_reproj_error_spread = json_config["RANSAC"]["RANSAC_reproj_error_spread"];

    // Other VO setup
    median_treshhold = json_config["vo"]["median_treshhold"];
    match_ratio_limit = json_config["vo"]["match_ratio_limit"];
    rotation_deg_threshold = json_config["vo"]["rotation_deg_threshold"];
    track_length_threshold = json_config["vo"]["track_length_threshold"];

}

Feature_Based_Visual_Odometry::Feature_Based_Visual_Odometry(
        detector_type type, 
        detector_configuration config,
        bool undistort,
        bool greyscale,
        cv::Size frame_size,
        double scale_factor,
        double aspect_minus_one, 
        cv::Mat camera_matrix,
        cv::Mat dist_coeffs,
        int RANSAC_tries,
        int RANSAC_iterations,
        double RANSAC_reproj_error,
        double RANSAC_reproj_error_spread,
        bool debug
        ) {

    // Debugging 
    this->debug = debug;

    // Camera and frame configuration setup
    this->undistort_frame = undistort;
    this->greyscale = greyscale;
    this->frame_size = frame_size;
    this->scale_factor = scale_factor;
    this->aspect_minus_one = aspect_minus_one;
    
    this->resized_frame_size = cv::Size(frame_size.width * scale_factor, frame_size.height * scale_factor);

    this->dist_coeffs = dist_coeffs;
    this->camera_matrix = camera_matrix;
    this->camera_matrix.at<double>(0,0) *= scale_factor; // fx, fl
    this->camera_matrix.at<double>(1,1) *= scale_factor; // fy, fl
    this->camera_matrix.at<double>(0,2) *= scale_factor; // cx, ppx
    this->camera_matrix.at<double>(1,2) *= scale_factor / (1 + aspect_minus_one); // cy, ppy
    this->new_camera_matrix = cv::getOptimalNewCameraMatrix(camera_matrix, dist_coeffs, resized_frame_size, 1, resized_frame_size, &frame_roi);

    detector_config = config;
    this->detector = i_detector::create(type, std::move(config));

    // RANSAC setup
    this->RANSAC_tries = RANSAC_tries;
    this->RANSAC_iterations = RANSAC_iterations;
    this->RANSAC_reproj_error = RANSAC_reproj_error;
    this->RANSAC_reproj_error_spread = RANSAC_reproj_error_spread;
} 


float Feature_Based_Visual_Odometry::calculate_median_distance(std::vector<cv::KeyPoint> &kpts_old, std::vector<cv::KeyPoint> &kpts_new, std::vector<cv::DMatch> &matches){
    std::vector<float> shifts;

    for (const auto& match : matches) {
        // cv::Point2f pt_old = kpts_old[match.trainIdx].pt; // wrong
        // cv::Point2f pt_new = kpts_new[match.queryIdx].pt;

        cv::Point2f pt_old = kpts_old[match.queryIdx].pt;
        cv::Point2f pt_new = kpts_new[match.trainIdx].pt;

        float dx = pt_new.x - pt_old.x;
        float dy = pt_new.y - pt_old.y;
        float distance = std::sqrt(dx * dx + dy * dy);

        shifts.push_back(distance);
    }

    std::sort(shifts.begin(), shifts.end());

    if (shifts.empty()) return 0.0f;

    size_t n = shifts.size();
    if (n % 2 == 0) {
        return (shifts[n / 2 - 1] + shifts[n / 2]) / 2.0f;
    } else {
        return shifts[n / 2];
    }
}

void Feature_Based_Visual_Odometry::get_matched_pts_and_desc(
    std::vector<cv::Point2f> &kpts_cords_old, std::vector<cv::Point2f> &kpts_cords_new,
    std::vector<cv::KeyPoint> &kpts_old, std::vector<cv::KeyPoint> &kpts_new,
    cv::Mat &desc_old, cv::Mat &desc_new
    ) {

    for (cv::DMatch match : detector->matches) {
        if (match.queryIdx >= detector->keypoints1.size() ||
            match.trainIdx >= detector->keypoints2.size() ||
            match.queryIdx >= detector->descriptors1.rows ||
            match.trainIdx >= detector->descriptors2.rows) {
            continue;  // Skip invalid matches. Matches are out of bound from the other array
        }

        kpts_cords_old.push_back(detector->keypoints1[match.queryIdx].pt);
        kpts_cords_new.push_back(detector->keypoints2[match.trainIdx].pt);

        kpts_old.push_back(detector->keypoints1[match.queryIdx]);
        kpts_new.push_back(detector->keypoints2[match.trainIdx]);
        
        // desc_old.push_back(descriptors1_clone.row(match.queryIdx));
        // desc_new.push_back(descriptors2_clone.row(match.trainIdx));

        desc_old.push_back(detector->descriptors1.row(match.queryIdx));
        desc_new.push_back(detector->descriptors2.row(match.trainIdx));
    }
}

void Feature_Based_Visual_Odometry::geometric_analysis(
    std::vector<cv::Point2f> &old_kpts_coordinates, 
    std::vector<cv::Point2f> &new_kpts_coordinates, 
    cv::Mat &E, 
    cv::Mat &mask
    ){
    E = cv::findEssentialMat(old_kpts_coordinates, new_kpts_coordinates, camera_matrix, cv::RANSAC, 0.999, 1.0, mask);
}

void Feature_Based_Visual_Odometry::calculate_relative_motion(cv::Mat &E,
    std::vector<cv::Point2f> &old_kpts_coordinates, std::vector<cv::Point2f> &new_kpts_coordinates, 
    cv::Mat &R, cv::Mat &t, cv::Mat &triangulated_points, cv::Mat &mask
    ){
    //double dist_tresh = 100;
    cv::recoverPose(E, old_kpts_coordinates, new_kpts_coordinates, camera_matrix, R, t, 100, mask, triangulated_points);
}

double Feature_Based_Visual_Odometry::calculate_reprojection_error(std::vector<cv::Point2f> points1, std::vector<cv::Point2f> points2, cv::Mat inliers) {
    double err_diff = 0.0;
    int num_inliers = inliers.rows;

    for (int i = 0; i < num_inliers; ++i) {
        int idx = inliers.at<int>(i, 0); 
        cv::Point2f diff = points1[idx] - points2[idx];
        err_diff += diff.x * diff.x + diff.y * diff.y;
    }
    if (num_inliers > 0) {
        return std::sqrt(err_diff / num_inliers);
    } else {
        return 0.0;
    }
}

bool Feature_Based_Visual_Odometry::rotation_check(cv::Mat rot){
    if (prev_rotations.size() > 0) {
        // get the relative rotation between the two frames
        cv::Mat rel_rotation = prev_rotations.back().inv() * rot;

        // convert rotation matrix into rotation vector
        cv::Mat rel_rotation_vect;
        cv::Rodrigues(rel_rotation, rel_rotation_vect);

        // calculate relative rotational angle 
        double rel_angle_rad = cv::norm(rel_rotation_vect);
        double rel_angle_deg = rel_angle_rad * 180.0 / CV_PI;

        // check against the threshold
        // std::cout << "angle: " << rel_angle_deg << std::endl;
        if (rel_angle_deg > rotation_deg_threshold) {
            return false;
        }
    }
    return true;
}

void Feature_Based_Visual_Odometry::plot_matches(cv::Mat &image, std::vector<cv::Point2f> &old_kpts_coordinates, std::vector<cv::Point2f> &new_kpts_coordinates,cv::Mat &mask){  
    for(int i = 0; i < mask.rows; i++) {
        for(int j = 0; j < mask.cols; j++) {
            if (mask.at<uchar>(i, j) == 1) {
                cv::line(image, old_kpts_coordinates[i], new_kpts_coordinates[i], cv::Scalar(255, 255, 255), 2);
                cv::circle(image, old_kpts_coordinates[i], 2, cv::Scalar(128, 128, 128), 2);
                cv::circle(image, new_kpts_coordinates[i], 2, cv::Scalar(0, 255, 255), 2);
            }
        }
    }
}

void Feature_Based_Visual_Odometry::run(cv::Mat frame_in){
    if (frame_in.empty()) {
        // std::cout << "Skip frame: frame empty" << std::endl;
        return;
    }
    int64_t frame_timestamp = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();
    std::cout << "frame: " << frame_n++ << ", time: " << frame_timestamp << std::endl;

    cv::Mat frame;
    track_length++;

    // Frame undistortion correction, rescaling and greyscaling
    // TODO: Look into gamma correction?
    if (undistort_frame) {
        cv::resize(frame_in, frame_in, resized_frame_size, 0.0f, 0.0f, cv::INTER_AREA);
        cv::undistort(frame_in, frame, camera_matrix, dist_coeffs, new_camera_matrix);
        frame = frame(frame_roi).clone();
        if (greyscale) {
            cv::cvtColor(frame, frame, cv::COLOR_BGR2GRAY);
        }
        
    } else {
        if (greyscale) {
            cv::cvtColor(frame_in, frame, cv::COLOR_BGR2GRAY);
        } else {
            frame = frame_in;
        }
        cv::resize(frame, frame, resized_frame_size, 0.0f, 0.0f, cv::INTER_AREA);
    }

    if (keyframes.size() == 0 || failed_track) {

        // First incoming frame
        // std::cout << "starting new\n";
        detector->image1 = frame;
        detector->detect_features_img1();
        keyframe new_kf = {
            frame_timestamp, 
            frame, 
            detector->keypoints1, 
            detector->descriptors1.clone(), 
            std::vector<cv::DMatch>(), 
            cv::Mat(), 
            cv::Mat(), 
            global_pos, 
            global_rot
        };
        keyframes.push_back(new_kf);

        // Reset flags and counters
        failed_track = false;
        scale_error = false;
        track_length = 0;

        
    } else {

        int64_t ts1 = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();

        // Take last keyframe
        keyframe last_kf = keyframes.back();
        detector->image1 = last_kf.frame; // not needed
        detector->keypoints1 = last_kf.keypoints;
        detector->descriptors1 = last_kf.descriptors;

        detector->image2 = frame;
        detector->detect_features_img2();
        if (detector->descriptors2.empty()) {
            std::cout << "Skipping frame. No features found" << std::endl;
            match_fail_counter++;
            if (match_fail_counter > match_fail_counter_threshold) {
                failed_track = true;
            }
            if (covariance < covariance_cap){
                covariance +=  0.0;
            }
            return;
        } else {
            match_fail_counter = 0;
        }

        int64_t ts2 = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();

        detector->match_features(); // what is the minimum required matches? DOF - 1 = 5?
        if (detector->matches.size() < 8) {
            // std::cout << "Skip frame: Too few matches" << std::endl;
            // TODO: Should this be used to take a new keyframe
            if (covariance < covariance_cap){
                covariance +=  0.0;
            }
            return;
        } 
        double cur_match_ratio =  (double)detector->matches_count / (double)detector->raw_matches_count;
        // std::cout << match_ratio << std::endl;

        int64_t ts3 = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();

        // Get keypoints, keypoint coordinates and descriptors according to the found matches.
        std::vector<cv::Point2f> kpts_cords_old, kpts_cords_new;
        std::vector<cv::KeyPoint> kpts_old, kpts_new;
        cv::Mat desc_old, desc_new;
        get_matched_pts_and_desc(kpts_cords_old, kpts_cords_new, kpts_old, kpts_new, desc_old, desc_new);

        // Pose estimation
        cv::Mat essential_mat, mask;
        geometric_analysis(kpts_cords_old, kpts_cords_new, essential_mat, mask);
        // Check that the essential matrix is valid
        // More info: https://answers.opencv.org/question/136092/findessentialmat-returns-3x30-or-3x12-mat/
        if (essential_mat.empty() || essential_mat.rows != 3 || essential_mat.cols != 3) {
            // std::cout << "Invalid Essential matrix: size is " << essential_mat.rows << "x" << essential_mat.cols << std::endl; // should this be cerr?
            std::cout << "Essential matrix:\n" << essential_mat << std::endl;
            return;
        }
        cv::Mat R, t, trig_points;
        calculate_relative_motion(essential_mat, kpts_cords_old, kpts_cords_new, R, t, trig_points, mask);
        
        cv::Mat points3d_mat;
        cv::convertPointsFromHomogeneous(trig_points.t(), points3d_mat);
        if (points3d_mat.rows < 10) {
            std::cout << "Skip frame: Too few 3d points" << std::endl;
            return;
        }

        // proposed 
        cv::Mat E_proposed, mask_proposed, t_proposed, R_proposed;
        cv::transpose(essential_mat, E_proposed);
        cv::multiply(E_proposed, -1, E_proposed);
        mask_proposed = mask;
        cv::multiply(t, -1, t_proposed);
        cv::transpose(R, R_proposed);

        if (debug) {
            try {
                cv::Mat matches_img = frame;
                cv::cvtColor(matches_img, matches_img, cv::COLOR_GRAY2BGR);
                plot_matches(matches_img, kpts_cords_old, kpts_cords_new, mask);
                cv::imshow("matches", matches_img);
                cv::waitKey(frame_time_ms);
            } catch (const std::exception& e) {
                // std::cerr << "Exception caught: " << e.what() << std::endl;
            }
        }

        // 
        cv::Mat rvec_guess, tvec_guess;
        cv::Rodrigues(R_proposed, rvec_guess); // Converting into rotation vector
        tvec_guess = t_proposed;
        //auto matches = detector->matches;

        // 3D to 2D motion estimation (PnP)
        // RANSAC go brr
        double best_err = DBL_MAX;
        cv::Mat best_tvec_guess, best_rvec_guess, best_inliers;
        std::vector<cv::Point2f> best_image_points;
        for (int i = 0; i < RANSAC_tries; i++) {
            cv::Mat inliers;
            cv::Mat obj_pts, img_pts;
            // TODO: Check the flags question
            std::vector<int> flags = {cv::SOLVEPNP_ITERATIVE, cv::SOLVEPNP_DLS, cv::SOLVEPNP_EPNP, cv::SOLVEPNP_P3P, cv::SOLVEPNP_UPNP, cv::SOLVEPNP_AP3P, cv::SOLVEPNP_ITERATIVE}; // Broken flags? for SOLVEPNP_DLS and SOLVEPNP_UPNP?
            int flag = flags[i % (flags.size())]; // loop through flags array
            bool use_extrinsic_guess = (i % 2 == 1); // toggle between true/false

            // Generate variation for reprojection error
            RANSAC_rng.state = i;
            double rng_noise = RANSAC_rng.uniform(-0.5, 0.5);
            double reproj_error = RANSAC_reproj_error - rng_noise * RANSAC_reproj_error_spread;

            // Perspective n point (PnP) pose computation
            cv::Mat tvec = tvec_guess.clone();
            cv::Mat rvec = rvec_guess.clone();
            bool success = cv::solvePnPRansac(points3d_mat, kpts_cords_new, camera_matrix, cv::Mat(), rvec, tvec, use_extrinsic_guess, 100, reproj_error, 0.99f, inliers, flag); // Should I leave distCoeffs empty?

            // Converting cv::Mat into vector nx3 vector. 
            // Reshaping function doesn't seem to work for reshaping
            // cv:Mat from 3 channel nx1 to 1 channel nx3 for cv::projectPoints
            // points3d_mat = points3d_mat.reshape(1);
            std::vector<cv::Point3f> points3d_vect;
            for (int i = 0; i < points3d_mat.rows; i++) {
                points3d_vect.push_back(cv::Point3f(
                    points3d_mat.at<double>(i, 0),
                    points3d_mat.at<double>(i, 1),
                    points3d_mat.at<double>(i, 2)
                ));
            }

            // If solution is found, then calculate point projections and reprojection error
            // Get the best solution
            if (success) {
                // Project 3D point coordinates onto 2D image points.
                std::vector<cv::Point2f> image_points;
                cv::Mat R_mat;
                cv::Rodrigues(rvec, R_mat); // Is R_mat needed? Can I just use rvec?
                cv::projectPoints(points3d_vect, R_mat, tvec, camera_matrix, cv::Mat(), image_points); // Should I pass empty matrix for no distortion?

                // Calculate reprojection error
                double err_diff = calculate_reprojection_error(image_points, kpts_cords_new, inliers);
                // std::cout << "Reprojection Error: " << err_diff << std::endl;

                // Get the best result
                if (best_err > err_diff) {
                    best_err = err_diff;
                    best_rvec_guess = rvec;
                    best_tvec_guess = tvec;
                    best_inliers = inliers;
                    best_image_points = image_points;
                }
                // if (best_err < 4) {  // TODO: threshold to break loop?
                //     //break;
                // }
            } else {
                // std::cerr << "\nFAILED TO GET SOLUTION\n" << std::endl;
            }
        }
        // std::cout << "Best reprojection Error: " << best_err << std::endl;

        cv::Mat rvec = best_rvec_guess;
        cv::Mat tvec = best_tvec_guess;

        cv::Mat R_proj, t_proj; 
        
        cv::Rodrigues(rvec, R_proj); // Converting into rotation vector
        t_proj = tvec;
        
        R = R_proj.t();
        t = -R * t_proj; // Negated dot product
        
        // TODO: Scale motion to real world values?
        cv::Mat R_scaled, t_scaled;
        R = scale_cm * R * scale_cm.t();
        t = scale_cm * t;
        // if (prev_rotations.size() > 100) {

        
            // if (!rotation_check(R)) {
            //     std::cout << "Bad rotation" << std::endl;


            //     // keyframe new_kf = {
            //     //     frame_timestamp, 
            //     //     frame.clone(), 
            //     //     detector->keypoints2, 
            //     //     detector->descriptors2.clone(), 
            //     //     std::vector<cv::DMatch>(), 
            //     //     cv::Mat(), 
            //     //     cv::Mat(),
            //     //     global_pos.clone(),
            //     //     global_rot.clone(),
            //     // };

            //     // keyframes.push_back(new_kf);

            //     // prev_rotations.push_back(R);
            //     // prev_translations.push_back(t);     

            //     scale_error = false;
            //     track_length = 0;

            //     if (covariance < covariance_cap){
            //         covariance +=  0.0;
            //     }

            //     // failed_track = true;

                
            // } 
        // }

        if (!failed_track) {
            prev_rotations.push_back(R);
            prev_translations.push_back(t);
        }


        // TODO: scaling check
        // if current frame is a outlier then take mean of the x last ones
        double scale = cv::norm(t);
        // std::cout << "scale: " << scale << std::endl;

        if (scales.size() > 50) {
            double mean_scale = std::accumulate(scales.end() - 30, scales.end(), 0.0) / 30.0;
            if (scale > mean_scale * 5) {
                std::cout << "Relative Scale too big Dropping frame 1" << std::endl;
                scale = mean_scale;
                scale_error = true;
            }
        }
        else if (scales.size() > 2) {
            if (scale > scales[scales.size() - 2] * 50) {
                // std::cout << "Scale: " << scale << std::endl;
                // std::cout << "Previous Scale: " << scales[scales.size() - 2] << std::endl;
                std::cout << "Relative Scale too big Dropping frame 2" << std::endl;
            }
        }

        cv::Mat pos_update = global_rot * (t * scale);
        cv::Mat pos = global_pos + pos_update;
        cv::Mat rot = global_rot * R;

        if (!failed_track) {
            scales.push_back(scale); 
            global_positions_vector.push_back(pos);
        }
    

        int64_t ts4 = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();

        float median = calculate_median_distance(last_kf.keypoints, detector->keypoints2, detector->matches);
        if (match_ratio < cur_match_ratio) {
            match_ratio = cur_match_ratio;
        }
        if ((median > median_treshhold || cur_match_ratio < match_ratio * match_ratio_limit || scale_error || track_length > track_length_threshold) && !failed_track) {
            std::cout << "Take new keyframe." << std::endl;
            // std::cout << (median > median_treshhold) << " " << (cur_match_ratio < match_ratio * match_ratio_limit) << " "  << scale_error << " " << (track_length > track_length_threshold) << std::endl;
            scale_error = false;
            track_length = 0;

            global_pos = pos; 
            global_rot = rot;

            keyframe new_kf = {
                frame_timestamp, 
                frame, 
                detector->keypoints2,
                detector->descriptors2.clone(), 
                detector->matches, 
                best_tvec_guess, 
                best_rvec_guess,
                pos.clone(),
                rot.clone()
            };
            keyframes.push_back(new_kf);
        }
        
        
        int64_t detection_time = ts2 - ts1;
        int64_t matching_time = ts3 - ts2;
        int64_t pose_est_time = ts4 - ts3;
        int64_t total_time = ts4 - frame_timestamp;

        detecting_ms_vect.push_back(detection_time);
        matching_ms_vect.push_back(matching_time);
        pose_est_ms_vect.push_back(pose_est_time);
        total_ms_vect.push_back(total_time);

        R.release();
        t.release();
        
    }

    (detector->image2).release();
    (detector->keypoints2).clear();
    (detector->descriptors2).release();
    (detector->matches).clear();
    std::cout << std::endl;
}