#include "detector_configurations.hpp"
#include "detector_wrappers.hpp"
#include <sstream>
#include <iostream>

using detector_configuration = std::variant<SIFT_CPU_config, SURF_CPU_config, ORB_CPU_config, AKAZE_CPU_config, ORB_GPU_config>;

void generate_configurations(detector_type type, std::vector<detector_configuration> &configurations){
    configurations.clear();
    switch (type) {
        // CUDA accelerated feature detectors

        case ORB_GPU:
            break;

        // CPU based feature detectors

        case SIFT_CPU:
            for (int i = 1000; i <= 8000; i += 1000){
                for (int j = 1; j <= 6; j++){
                    SIFT_CPU_config sift_cfg;
                    sift_cfg.nfeatures = i;
                    sift_cfg.nOctaveLayers = j;
                    configurations.push_back(sift_cfg);
                }
            }
            break;

        case SURF_CPU:
            for (double i = 50; i <= 400; i += 50) {
                for (int j = 1; j <= 6; j++) {
                    for (int k = 1; k <= 6; k++) {
                        SURF_CPU_config surf_cfg;
                        surf_cfg.hessianThreshold = i;
                        surf_cfg.nOctaveLayers = j;
                        surf_cfg.nOctaves = k;
                        surf_cfg.extended = true;
                        configurations.push_back(surf_cfg);
                    }
                }
            }
            break;

        case ORB_CPU:
            for (int i = 500; i <= 8000; i += 500){
                ORB_CPU_config orb_cfg;
                orb_cfg.nfeatures = i;
                configurations.push_back(orb_cfg);
            }
            break;

        case AKAZE_CPU:
            // Which parameters to test here?
            break;

        default:
            std::cout << "No configurations generated./n";
            break;
    }
}


std::string get_parameters_string(const detector_configuration &config) {
    return std::visit([](const auto& cfg) -> std::string {
        std::ostringstream param_stream;

        if constexpr (std::is_same_v<std::decay_t<decltype(cfg)>, SIFT_CPU_config>) {
            param_stream << "nfeatures=" << cfg.nfeatures
                         << ";nOctaveLayers=" << cfg.nOctaveLayers
                         << ";contrastThreshold=" << cfg.contrastThreshold
                         << ";edgeThreshold=" << cfg.edgeThreshold
                         << ";sigma=" << cfg.sigma;
        }
        else if constexpr (std::is_same_v<std::decay_t<decltype(cfg)>, SURF_CPU_config>) {
            param_stream << "hessianThreshold=" << cfg.hessianThreshold
                         << ";nOctaves=" << cfg.nOctaves
                         << ";nOctaveLayers=" << cfg.nOctaveLayers
                         << ";extended=" << (cfg.extended ? "true" : "false")
                         << ";upright=" << (cfg.upright ? "true" : "false");
        }
        else if constexpr (std::is_same_v<std::decay_t<decltype(cfg)>, ORB_CPU_config>) {
            param_stream << "nfeatures=" << cfg.nfeatures
                         << ";scaleFactor=" << cfg.scaleFactor
                         << ";nlevels=" << cfg.nlevels
                         << ";edgeThreshold=" << cfg.edgeThreshold
                         << ";firstLevel=" << cfg.firstLevel
                         << ";WTA_K=" << cfg.WTA_K
                         << ";scoreType=" << cfg.scoreType
                         << ";patchSize=" << cfg.patchSize
                         << ";fastThreshold=" << cfg.fastThreshold;
        }
        else if constexpr (std::is_same_v<std::decay_t<decltype(cfg)>, AKAZE_CPU_config>) {
            param_stream << "descriptor_type=" << cfg.descriptor_type
                         << ";descriptor_size=" << cfg.descriptor_size
                         << ";descriptor_channels=" << cfg.descriptor_channels
                         << ";threshold=" << cfg.threshold
                         << ";nOctaves=" << cfg.nOctaves
                         << ";nOctaveLayers=" << cfg.nOctaveLayers
                         << ";diffusivity=" << cfg.diffusivity
                         << ";max_points=" << cfg.max_points;
        }
        else if constexpr (std::is_same_v<std::decay_t<decltype(cfg)>, ORB_GPU_config>) {
            param_stream << "nfeatures=" << cfg.nfeatures
                         << ";scaleFactor=" << cfg.scaleFactor
                         << ";nlevels=" << cfg.nlevels
                         << ";edgeThreshold=" << cfg.edgeThreshold
                         << ";firstLevel=" << cfg.firstLevel
                         << ";WTA_K=" << cfg.WTA_K
                         << ";scoreType=" << cfg.scoreType
                         << ";patchSize=" << cfg.patchSize
                         << ";fastThreshold=" << cfg.fastThreshold
                         << ";blurForDescriptor=" << (cfg.blurForDescriptor ? "true" : "false");
        }
        else {
            param_stream << "Unknown Config";
        }

        return param_stream.str();
    }, config);
}


