#pragma once

#include <vector>

#include <opencv2/opencv.hpp>
#include <opencv2/core.hpp>

#include "detector_wrappers.hpp"
#include "detector_configurations.hpp"



struct keyframe {
    int64_t timestamp_ms;
    cv::Mat frame;
    std::vector<cv::KeyPoint> keypoints;
    cv::Mat descriptors;
    std::vector<cv::DMatch> matches; // matches to the previous keyframe
    cv::Mat translation;
    cv::Mat rotation;
    cv::Mat global_pos;
    cv::Mat global_rot;

    // Need to look through what data types are needed for each variable in keyframe
};

class Feature_Based_Visual_Odometry {
    public:
        // Camera stuff
        cv::Size frame_size = cv::Size(1280, 720); // Default camera frame size for Kuup cv::Size(1280, 720) KITTI: cv::Size(1241, 376)
        cv::Size resized_frame_size = cv::Size(640, 360); // for downscaling the frame in piepline
        bool undistort_frame = true;
        bool greyscale = true;
        double scale_factor = 1;
        double aspect_minus_one = 0;
        // cv::Mat camera_matrix = cv::Mat_<double>(3, 3);
        cv::Mat camera_matrix = (cv::Mat_<double>(3, 3) << 
                                811.27847985, 0, 511.5733993,
                                0, 806.82258429, 295.02167692,
                                0, 0, 1);
        cv::Mat new_camera_matrix;
        // cv::Mat dist_coeffs = cv::Mat_<double>(1, 5);
        cv::Mat dist_coeffs = (cv::Mat_<double>(1, 5) << -0.43500715, 0.28144541, 0.00110741, -0.00460803, -0.13176509);
        //double ppx, ppy; // Principal point coordinates
        //cv::Mat camera_map_x, camera_map_y;
        cv::Rect frame_roi;
        //int roi_x, roi_y, roi_w, roi_h;

        // Detector stuff
        std::shared_ptr<i_detector> detector = i_detector::create(detector_type::SIFT_CPU ,detector_configuration{SIFT_CPU_config{}});;
        detector_configuration detector_config;
        float lowes_ratio = 0.8; // I think 0.8 was used in Lowe's paper
        
        // Keyframes stuff
        std::vector<keyframe> keyframes;
        double median_treshhold = 35;
        int frame_n = 0;
        double match_ratio = 0;
        double match_ratio_limit = 0.4;
        bool scale_error = false;
        int track_length = 0;
        int track_length_threshold = 0;
        int match_fail_counter = 0;
        int match_fail_counter_threshold = 5;
        bool failed_track = false;

        // Ransac stuff
        int RANSAC_tries = 10;
        int RANSAC_iterations = 100;
        int RANSAC_reproj_error = 8.0f;
        int RANSAC_reproj_error_spread = 2.0f;
        std::vector<int> RANSAC_flags = {cv::SOLVEPNP_ITERATIVE, cv::SOLVEPNP_EPNP, cv::SOLVEPNP_P3P, cv::SOLVEPNP_AP3P, cv::SOLVEPNP_ITERATIVE}; // cv::SOLVEPNP_DLS, cv::SOLVEPNP_UPNP are broken
        cv::RNG RANSAC_rng;

        // Motion stuff 
        std::vector<double> scales;
        std::vector<cv::Mat> prev_rotations, prev_translations;
        double rotation_deg_threshold = 5;  
        cv::Mat global_pos = cv::Mat::zeros(3, 1, CV_64F);
        cv::Mat global_rot = cv::Mat::eye(3, 3, CV_64F); 
        cv::Mat scale_cm = (cv::Mat_<double>(3, 3) << 
            3.33066907e-16, 0.00000000e+00, 1.00000000e+00,
            -1.00000000e+00, 3.33066907e-16, 0.00000000e+00,
            0.00000000e+00, -1.00000000e+00, 3.33066907e-16
            );

        double covariance = 0.3; // What should this be initsialised at?
        double covariance_cap = 5;

        // Debugging stuff
        bool debug = false;
        std::vector<cv::Mat> global_positions_vector;
        int frame_time_ms = 0;
        std::vector<int64_t> detecting_ms_vect, matching_ms_vect, pose_est_ms_vect, total_ms_vect;
        void plot_matches(cv::Mat &image, std::vector<cv::Point2f> &old_kpts_coordinates, std::vector<cv::Point2f> &new_kpts_coordinates,cv::Mat &mask);

        Feature_Based_Visual_Odometry();
        Feature_Based_Visual_Odometry(json json_config);
        Feature_Based_Visual_Odometry(
            detector_type type, 
            detector_configuration config,
            bool undistort,
            bool greyscale,
            cv::Size frame_size,
            double scale_factor,
            double aspect_minus_one, 
            cv::Mat camera_matrix,
            cv::Mat dist_coeffs,
            int RANSAC_tries,
            int RANSAC_iterations,
            double RANSAC_reproj_error,
            double RANSAC_reproj_error_spread,
            bool debug
            );
        
        
        void run(cv::Mat frame);
        void get_matched_pts_and_desc(
            std::vector<cv::Point2f> &kpts_cords_old, std::vector<cv::Point2f> &kpts_cords_new,
            std::vector<cv::KeyPoint> &kpts_old, std::vector<cv::KeyPoint> &kpts_new,
            cv::Mat &desc_old, cv::Mat &desc_new
            );
        float calculate_median_distance(std::vector<cv::KeyPoint> &kpts_old, std::vector<cv::KeyPoint> &kpts_new, std::vector<cv::DMatch> &matches);
        void geometric_analysis(std::vector<cv::Point2f> &old_kpts_coordinates, std::vector<cv::Point2f> &new_kpts_coordinates, cv::Mat &E, cv::Mat &mask);
        void calculate_relative_motion(cv::Mat &E, std::vector<cv::Point2f> &old_kpts_coordinates, std::vector<cv::Point2f> &new_kpts_coordinates, cv::Mat &R, cv::Mat &t, cv::Mat &triangulated_points, cv::Mat &mask);
        double calculate_reprojection_error(std::vector<cv::Point2f> points1, std::vector<cv::Point2f> points2, cv::Mat inliers);
        bool rotation_check(cv::Mat rot);
    };