#pragma once

#include <opencv2/opencv.hpp>
#include <opencv2/core.hpp>
#include <opencv2/features2d.hpp>  // For SIFT, FAST, ORB
#include <opencv2/xfeatures2d.hpp> // For SURF, STAR, MSD, HLF, GFFT, BRIEF?
#include <opencv2/cudafeatures2d.hpp> // For CUDA accelerated: FAST, ORB

#include <nlohmann/json.hpp>
using json = nlohmann::json;

#include "detector_configurations.hpp"

enum detector_type {
    // CPU-based detectors
    SIFT_CPU, 
    SURF_CPU, 
    ORB_CPU, 
    AKAZE_CPU, 
    // CUDA accelerated detectors
    ORB_GPU 
};

using detector_configuration = std::variant<SIFT_CPU_config, SURF_CPU_config, ORB_CPU_config, AKAZE_CPU_config, ORB_GPU_config>;


// General detector interface class
class i_detector {
    public:
        detector_type type;
        cv::Mat image1, image2, descriptors1, descriptors2;
        std::vector<cv::KeyPoint> keypoints1, keypoints2;
        std::vector<cv::DMatch> matches;
        float lowes_ratio_thresh = 0.8f;
        int raw_matches_count, matches_count;
    
        virtual void detect_features_img1() = 0;
        virtual void detect_features_img2() = 0;
        virtual void match_features() = 0;
        virtual ~i_detector() = default;
        void show_features();
        template<typename T> T get_parameter_with_default(const json& config, const std::string& parameter, const T& default_value);
        template<typename T> T get_parameter_with_default(const json &config, const std::unordered_map<std::string, T> &score_type_map, const std::string &parameter, const T& default_value);
    
        // Factory method declaration
        static std::unique_ptr<i_detector> create(detector_type type, detector_configuration &&config);
        static std::unique_ptr<i_detector> create(json json_config);
    };


// ORB CPU detector
class orb_cpu_detector : public i_detector {
    public:
        cv::Ptr<cv::Feature2D> detector;
        cv::Ptr<cv::BFMatcher> matcher;

        orb_cpu_detector(detector_configuration &&config);
        orb_cpu_detector(json config);
        void detect_features_img1() override;
        void detect_features_img2() override;
        void match_features() override;
    };


// SIFT CPU detector
class sift_cpu_detector : public i_detector {
    public:
        cv::Ptr<cv::Feature2D> detector;
        cv::Ptr<cv::FlannBasedMatcher> matcher;

        sift_cpu_detector(detector_configuration &&config);
        sift_cpu_detector(json config);
        void detect_features_img1() override;
        void detect_features_img2() override;
        void match_features() override;
    };


// SURF CPU detector
class surf_cpu_detector : public i_detector {
    public:
        cv::Ptr<cv::Feature2D> detector;
        cv::Ptr<cv::FlannBasedMatcher> matcher;

        surf_cpu_detector(detector_configuration &&config);
        surf_cpu_detector(json config);
        void detect_features_img1() override;
        void detect_features_img2() override;
        void match_features() override;
    };


// AKAZE CPU detector
class akaze_cpu_detector : public i_detector {
    public:
        cv::Ptr<cv::Feature2D> detector;
        cv::Ptr<cv::BFMatcher> matcher;

        akaze_cpu_detector(detector_configuration &&config);
        akaze_cpu_detector(json config);
        void detect_features_img1() override;
        void detect_features_img2() override;
        void match_features() override;
    };


// ORB CUDA accelerated detector
class orb_cuda_detector : public i_detector {
    public:
        cv::cuda::GpuMat d_image1, d_keypoints1, d_descriptors1, d_image2, d_keypoints2, d_descriptors2;
        cv::Ptr<cv::cuda::ORB> detector;
        cv::Ptr<cv::cuda::DescriptorMatcher> matcher;

        orb_cuda_detector(detector_configuration &&config);
        orb_cuda_detector(json config);
        void detect_features_img1() override;
        void detect_features_img2() override;
        void match_features() override;
    };
