#include <vector>
#include <iostream>
#include <fstream>
#include <sstream>

#include <opencv2/opencv.hpp>
#include <opencv2/core.hpp>

#include "feature_based_vo.hpp"

// logging 
#include <uprofile.h>
#include <igpumonitor.h>
#include <nvidiamonitor.h> 


using namespace std;
using namespace cv;


void loop_thru_frames(std::string json_config_path, std::string img_directory, std::string out_csv_directory) {
    // uprofile::addGPUMonitor(new uprofile::NvidiaMonitor());

    std::vector<cv::String> fn;
    cv::glob(img_directory + "*.png", fn, false);
    if (fn.empty()) {
        cv::glob(img_directory + "*.jpg", fn, false);
    }
    if (fn.empty()) {
        std::cerr << "No images found in " << img_directory << std::endl;
        return;
    } else {
        std::cout << "Images found ("  << img_directory << "): " << fn.size() << endl;
    }

    std::ifstream f(json_config_path);
    json json_config = json::parse(f);

    std::cout << "Using VO configuration: "  << json_config_path << std::endl;
    auto VO = Feature_Based_Visual_Odometry(json_config);

    // for (size_t i = 450; i < 900; i++) {
    for (size_t i = 1020; i < fn.size(); i++) {
        cv::Mat image = cv::imread(fn[i]);
        VO.run(image);
    }

    // Extracting frame translations and rotations for motion plotting
    auto frames = VO.keyframes;
    std::ofstream csv_file_t, csv_file_r, csv_file_pos;
    csv_file_t.open(out_csv_directory + "_translation.csv");
    csv_file_r.open(out_csv_directory + "_rotation.csv");
    csv_file_pos.open(out_csv_directory + "_positions.csv");

    // frames.erase(frames.begin()); // delete the first frame without matches
    // for (const keyframe &frame : frames) {
    //     csv_file_t << frame.translation.at<double>(0, 0) << ";" 
    //                << frame.translation.at<double>(1, 0) << ";"  
    //                << frame.translation.at<double>(2, 0) << std::endl;

    //     csv_file_r << frame.rotation.at<double>(0, 0) << ";"
    //                << frame.rotation.at<double>(1, 0) << ";"
    //                << frame.rotation.at<double>(2, 0) << std::endl;

    //     // csv_file_pos << frame.global_pos.at<double>(0, 0) << ";"
    //     //             << frame.global_pos.at<double>(1, 0) << ";"
    //     //             << frame.global_pos.at<double>(2, 0) << std::endl;
    //             }
    
    for (auto pos : VO.global_positions_vector) {
        csv_file_pos << pos.at<double>(0, 0) << ";"
                    << pos.at<double>(1, 0) << ";"
                    << pos.at<double>(2, 0) << std::endl;
    }

    for (auto pos : VO.prev_translations) {
        csv_file_t << pos.at<double>(0, 0) << ";"
                    << pos.at<double>(1, 0) << ";"
                    << pos.at<double>(2, 0) << std::endl;
    }

    csv_file_t.close();
    csv_file_r.close();
    csv_file_pos.close();
    std::cout << "file" << std::endl;
}


void test_vo(json json_config){
    std::vector<cv::String> fn;
    std::string img_directory = json_config["img_directory"];
    cv::glob(img_directory + "*.png", fn, false);
    if (fn.empty()) {
        cv::glob(img_directory + "*.jpg", fn, false);
    }
    if (fn.empty()) {
        std::cerr << "No images found in " << img_directory << std::endl;
        return;
    } else {
        std::cout << "Images found ("  << img_directory << "): " << fn.size() << endl;
    }

    std::ofstream test_times_csv;
    test_times_csv.open("test_times.csv");
    test_times_csv << "detector" << ";" << "scale factor" << ";" << 
    "average detection time (ms)" << ";" << "detetion standard deviation" << ";" << 
    "average matching time (ms)" << ";" << "matching standard deviation"  << ";" << 
    "average pose est time (ms)" << ";" << "pose standard deviation" << ";" <<
    "average total time (ms)" << ";" << "total standard deviation" << std::endl;

    std::vector<std::string> detectors_vect = json_config["detectors"];
    std::vector<double> scale_factors_vect = json_config["scale_factors"];

    for (std::string detector : detectors_vect) {
        for (double scale_factor : scale_factors_vect) {
            std::cout << "Starting test for " << detector << " with scale factor " << scale_factor <<  std::endl;
            // Start recording metrics to a log file
            uprofile::addGPUMonitor(new uprofile::NvidiaMonitor());
            uprofile::start((detector + "_" + std::to_string(scale_factor) + "_test_log.txt").c_str());
            uprofile::startCPUUsageMonitoring(100);
            uprofile::startProcessMemoryMonitoring (100);
            uprofile::startSystemMemoryMonitoring(100);
            uprofile::startGPUUsageMonitoring(100);
            uprofile::startGPUMemoryMonitoring(100);

            json_config["feature_detector"]["type"] = detector;
            json_config["camera"]["scale_factor"] = scale_factor;


            auto VO = new Feature_Based_Visual_Odometry(json_config);
            for (size_t i = 0; i < 2000; i++) {
                cv::Mat image = cv::imread(fn[i]);
                VO->run(image);
            }


            uprofile::stop();
            uprofile::removeGPUMonitor();

            // feature detection
            double detecting_ms_avg = std::reduce(VO->detecting_ms_vect.begin(), VO->detecting_ms_vect.end()) / VO->detecting_ms_vect.size();

            double sum_squared_diff = 0;
            for (auto val : VO->detecting_ms_vect) {
                double diff = val - detecting_ms_avg;
                sum_squared_diff += diff * diff;
            }
            double detecting_ms_std_dev = std::sqrt(sum_squared_diff / VO->detecting_ms_vect.size());


            // feature detection
            double matching_ms_avg = std::reduce(VO->matching_ms_vect.begin(), VO->matching_ms_vect.end()) / VO->matching_ms_vect.size();

            double sum_squared_diff_matching = 0;
            for (auto val : VO->matching_ms_vect) {
                double diff = val - matching_ms_avg;
                sum_squared_diff_matching += diff * diff;
            }
            double matching_ms_std_dev = std::sqrt(sum_squared_diff_matching / VO->matching_ms_vect.size());


            // pose estimation
            double pose_est_ms_avg = std::reduce(VO->pose_est_ms_vect.begin(), VO->pose_est_ms_vect.end()) / VO->pose_est_ms_vect.size();

            double sum_squared_diff_pose = 0;
            for (auto val : VO->pose_est_ms_vect) {
                double diff = val - pose_est_ms_avg;
                sum_squared_diff_pose += diff * diff;
            }
            double pose_est_ms_std_dev = std::sqrt(sum_squared_diff_pose / VO->pose_est_ms_vect.size());


            // Total time
            double total_ms_avg = std::reduce(VO->total_ms_vect.begin(), VO->total_ms_vect.end()) / VO->total_ms_vect.size();

            double sum_squared_diff_total = 0;
            for (auto val : VO->total_ms_vect) {
                double diff = val - total_ms_avg;
                sum_squared_diff_total += diff * diff;
            }
            double total_ms_std_dev = std::sqrt(sum_squared_diff_total / VO->total_ms_vect.size());



            test_times_csv << detector << ";" << scale_factor << ";" << 
            detecting_ms_avg << ";" <<  detecting_ms_std_dev << ";" << 
            matching_ms_avg << ";" <<  matching_ms_std_dev << ";" << 
            pose_est_ms_avg << ";" <<  pose_est_ms_std_dev << ";" <<
            total_ms_avg << ";" << total_ms_std_dev << std::endl;

            delete VO;
            std::cout << "Test complete" << std::endl;
        }
    }   
    test_times_csv.close();
}