import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Vector;


public class BoardStatus {
	public static final int STATUS_UNKNOWN = 0;
	public static final int STATUS_OK = 1;
	public static final int STATUS_STUCK = 2;
	public static final int STATUS_HAD_RESET = 3;
	public static final int STATUS_NOT_RESPONDING = 4;

	// one of STATUS_OK, STATUS_STUCK, STATUS_HAD_RESET, STATUS_NOT_RESPONDING
	int status;
	boolean stuck; 
	boolean allActionsComplete;
	
	int boardID;
	int motor0Coord;
	int maxCoord;
	int time;
	Date refreshTime;
	boolean refreshed;
	
	
	
	public static LinkedList allBoards;
	
	static {
		allBoards = new LinkedList();
	}
	
	public static void main(String[] args) {
		Main.main(args);
	}
	
	public static void printAll() {
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus brd = (BoardStatus) iter.next();
			System.out.println("Board nr " + brd.boardID);
		}
	}
	
	public BoardStatus(int brdId) {
		status = STATUS_UNKNOWN;
		boardID = brdId;
		refreshed = false;
		refreshTime = Calendar.getInstance().getTime();
	}
	
	/**
	 * just to create all boards, for motors that are in config... 
	 * when motor is read from config, this method is called and if no board 
	 * for this motor exists, appropriate board is created. 
	 */
	public static void accountMotor(int motorId) {
		BoardStatus bs = new BoardStatus(motorId);
		if (!allBoards.contains(bs)) {
			allBoards.add(bs);
		}
	}
	
	/**
	 * Creates position that contains current states for all motors.
	 * @param name the name to call this position, given from outside, probably determined by user input.
	 * @return Position object that contains current state of all the motors.
	 */
	public static Position getPosition(String name) {
		Position pos = new Position();
		pos.name = name;
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus brd = (BoardStatus) iter.next();
			pos.addMotorState(brd.boardID, brd.motor0Coord);
		}
		return pos;
	}
	
	/**
	 * called when status is received from some board over COM port.
	 * @param brdId boardID
	 * @param m0 first motor position
	 * @param m1 second motor position
	 * @param tm time of the board
	 */
	public static void refresh(int brdId, int m0, int maxCoord, int tm, boolean stuck, boolean complete) {
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus bs = (BoardStatus) iter.next();
			if (bs.boardID == brdId) {
				bs.stuck = stuck;
				bs.allActionsComplete = complete;
				bs.maxCoord = maxCoord;
				// this is the boardStatus for board that has sent this info 
				bs.checkReset(tm);
				
				bs.motor0Coord = m0;
				bs.time = tm;
				bs.refreshed = true;
				bs.refreshTime = Calendar.getInstance().getTime();
				if(!stuck) bs.status=STATUS_OK;
				else if(stuck) bs.status=STATUS_STUCK;
				
				System.out.println("##############");
				System.out.println("Status of board "+ bs.boardID);
				System.out.println("Stuck: " + bs.stuck);
				System.out.println("Complete: " + bs.allActionsComplete);
				System.out.println("Coord: " + bs.motor0Coord);
				System.out.println("MaxCoord: " + bs.maxCoord);
				System.out.println("Time: " + bs.time);
				System.out.println("Status: " + bs.status);
				System.out.println("##############");
			}
		}
	}
	
	/**
	 * This stuff requests statuses from all the boards, and waits for answer.
	 */
	public static void requestAllStatuses() {
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus brd = (BoardStatus) iter.next();
			MotorDriver.defaultDriver.sendStatusRequest(brd.boardID);
			System.out.println("polling status from board nr " + brd.boardID);
			Date time = Calendar.getInstance().getTime();
			for (int i = 0; (i < 7) && (brd.refreshTime.before(time)); i ++) {
				try {
					Thread.sleep(50);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			
			if (brd.refreshTime.before(time)) {
				brd.status = STATUS_NOT_RESPONDING;
				System.out.println("Board nr " + brd.boardID + " is not responding to status request");
			}
		}
	}
	
	public static void resetAllStuck() {
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus brd = (BoardStatus) iter.next();
			if (brd.status == STATUS_STUCK) {
				MotorDriver.defaultDriver.sendReset(brd.boardID);
			}
		}
	}
	
	public static Vector<String> displayMotorErrorSummary() {
		Vector<String> ret = new Vector();
		for (Iterator iter = allBoards.iterator(); iter.hasNext();) {
			BoardStatus brd = (BoardStatus) iter.next();
			switch (brd.status) {
				case STATUS_STUCK:
					ret.add("Motor nr. " + brd.boardID + " IS STUCK");
					break;
				case STATUS_UNKNOWN:
					ret.add("Motor nr. " + brd.boardID + " status is UNKNOWN");
					break;
				case STATUS_HAD_RESET:
					ret.add("Motor nr. " + brd.boardID + " HAD RESET");
					break;
				case STATUS_NOT_RESPONDING:
					ret.add("Motor nr. " + brd.boardID + " IS NOT RESPONDING");
					break;
			}
		}
		return ret;
	}
	
	/**
	 * Checks if reset has occured, by determining how mutch has board time 
	 * changed over time since last update.
	 * @param tm
	 */
	private void checkReset(int tm) {
		if (!refreshed) {
			return;
		}
		
		int deltaBrdTime = tm - time;
		long deltaRealTime = Calendar.getInstance().getTime().getTime() - refreshTime.getTime();
		long error = deltaRealTime - deltaBrdTime;
		float relation = ((float)deltaBrdTime) / ((float)deltaRealTime);
		
		if ((Math.abs(error) < 500) || ((relation < 1.01f) && (relation > 0.99f))) {
			// ok
		} else {
			status = STATUS_HAD_RESET;
			System.out.println("probably reset on board " + boardID + ": time difference is " + error);
			// reset has occured since last refresh...
			// TODO!!
		}
	}

	public boolean equals(Object obj) {
		if (obj instanceof BoardStatus) {
			return ((BoardStatus)obj).boardID == this.boardID;
		}
		return false;
	}
}
