#ifndef _UNIT_
#define _UNIT_

#include <assert.h>
#include <stdexcept>
#include <list>
#include "energy.h"

class UNIT;
class MOLECULE;
typedef enum{OK='O', GEOMFAIL='G', ENERGYFAIL='E'} Failcode;
typedef struct{double vdw, coulomb;} VdwCoulombPair;

#include "atom.h"
#include "chain.h"
#include "unitdef.h"
#include "bondlist.h"
#include "textfile.h"
#include "units.h"
#include "chains.h"
#include "unitid.h"
#include "genxyz.h"
#include "mcgen.h"
#include "bondinst.h"

using namespace std;

#define UNIT_PLACE_OK 0
#define CHAIN_IS_COMPLETED 1
#define UNIT_PLACE_EMPTY_UNIT 2
#define UNIT_PLACE_MAXTRYUNIT_EXCEEDED 3
#define UNIT_PLACE_RANDOM_TAKEBACK 4

#define HISTSIZE_UNKNOWN -1

#define TAKEBACK_STEPS_COUNT 1

#define UC_BEGIN     0x01 // BEGINUNIT   +
#define UC_REPEAT    0x02 // REPEATUNIT  + These are the normal units.
#define UC_CROSS     0x04 // CROSSUNIT   + 
#define UC_END       0x08 // ENDUNIT     +
#define UC_SEED      0x10 // virtual seedunit
#define UC_TAIL     0x100 // virtual tailunit
#define UC_PRESET    0x20 // This unit is fixed.
#define UC_INVISIBLE 0x40 // This unit is fixed, but atoms may be thrown out.
#define UC_EMPTY     0x80 // Unit without atoms. Empty placeholder.
#define UC_NORMAL   (UC_BEGIN|UC_REPEAT|UC_CROSS|UC_END)
#define UC_REAL     (UC_NORMAL|UC_PRESET|UC_INVISIBLE)

#define LONGEST_BACKREF 10 // units with unitid<LONGEST_BACKREF will not be
			   // possible to specify any bonds with.

class UNIT //! The working information for one unit to be used while generating.
{
private:

int atoms;
ATOM *atom;
CHAIN *chain;
CONNECT *connectfirst;
//int fieldconnectshift;
UNIT *previousunit, *nextunit, *sideunit;       // these properties connect
                                // all units into a tree.
bool is_sidechain; // this unit is sidechain for the previous unit.
//bool is_real; // shows if this unit contains real atoms
              // (contrary of being a virtual seed unit)
int unitclass; // Bitmask of UC_xxx. Shows which kind of unit it is.
bool has_coords;
int randstart;
        // When this variable counts down to zero, a random number of units
        // will be taken back.
long int totaltried;
        // Records how many times this unit has been tried to place since the
        // execution of mcgen.

MOLECULE *molecule; // parent molecule that this chain belongs to.

void init();    // Initialize the unit. (used by all constructors)
int set_previousunit_relation(UNIT *_previousunit, bool _is_sidechain);
        // creates a double-sided link between this and previous unit.
ENERGY energy, energy_cumul;
        // Declarations of unit energy variables
UNIT *previous_in_time; // these two properties connect all real (non-virtual)
static UNIT *last_in_time;// units into a linear (non-branching) linked list.
static vector<UNIT*> inited_units;
static list<UNIT*> units_outputting_order;
static bool units_outputting_order_ready;
int unit_realid;
int distance_from_last_cross;
int count_of_crosses;
int distance_from_chainbegin;
UNITDEF *unitdef;
static long unsigned int non_dumped_counter;
int histsize_before; // atoms history size before placing of this unit.
UNITID unitid;

int stat_count_geomfail;
double stat_minimal_energy;

static int init_units_outputting_order_recursion(UNIT *_where);
static long unsigned int best_dumped_size;
	// If this grows, then a new best set of dumpfiles will be written.
int last_atom;
const CROSS_RECORD * cross;
static vector<UNIT*> paused_units; //! List of units that recursive initialization skipped.
bool totally_internal; //! Becomes false if relative_unit() goes outside of a chain.

public:

int set_atoms(int _atoms);
        // Change size and allocate memory for the atom[] array
int get_atoms();
        // Returns the number of atoms in the atom[] array
ATOM *get_atom(int _atomid);
        // Return pointer to the _atomid'th atom of the unit
UNIT *relative_unit(int _unitid);
        // Return pointer to such unit that has given unitid relative to the
        // current unit
UNIT *find_unit_from_tree(int _chainid, int _unitid);
ATOM *relative_atom(int _chainid, int _unitid, int _atomid, bool _only_internal_bonds);
ATOM *relative_atom(FOUR_ATOMREF &_atomref, int _index, bool _only_internal_bonds);
bool overlaps_inside_unit();
bool overlaps_with(UNIT &_unit2);
bool not_overlapping();
        // Decide whether the molecule is geometrically correct in the sense
        // of atomic overlappings or not.
bool geomcheck_is_ok();
        // Decide whether the molecule is geometrically correct or not.
int log_unit_coords();
        // Write coordinates of all valid atoms in the current unit
        // into logfile.
        // Returns the count of printed atoms.
const string get_position();
        // Returns string of chainid and units position in that chain.
int print_defname();
        // Prints out which kind of unit it is.
        // For example C3 for CROSSUNIT with unitid 3.
string get_defname();
        // Returns short string that shows which kind of unit it is.
        // For example C3 for CROSSUNIT with unitid 3.
int print_structure_recursive();
int print_structure();
static int take_back_until(int _finalsize);
UNIT(UNIT *_previousunit, bool _is_sidechain);
        // Constructor for normal units
int gen_unique_unitid();
int choose_unitdef();
int choose_chain();
explicit UNIT(CHAIN *_chain);
        // Constructor for the virtual seed unit
explicit UNIT(const string _xyz_filename);
        // Constructor for units that are given as xyz-files.
UNIT();
        // Constructor for restoring dumped units
~UNIT();
        // Destructor
int calculate_distances();
int recursive_initialization();
static int initialize_paused_units();
static int choose_fieldconnects(MOLECULE *_molecule);
static int initialize_atoms_placement_order();
static int place_linear(MOLECULE *_molecule);
int place();
UNIT *get_previousunit() const;

private:

int place_seed();
        // Generate random coordinates for three virtual atoms
static int write_coords(const OUTPUTFILENAMES &_filenames);
        // This method writes all valid coordinates of the atoms in all units
        // into a xyz-file.
	// 
public:

static int count_all_placed_atoms();
        // This method counts all such atoms that have valid coordinates
        // (has_coords==true). Search is done over all units in the molecul.

bool fieldenergy_is_computable(CONNECT *_connect, bool _verbose);
	// Check whether all of bonds, angles and dihedrals are computable
bool bondenergy_is_computable(BONDLIST &_bondarray, uint _bondatomsnum,
		bool _verbose);
	// Check whether the calculation of bond, angle or dihedral energy
	// is free of atom pointing errors (mainly wrong chainid for a unitid.

//double calculate_energy();
static int recalculate_energy();
	// Recalculates energies of all units that have has_coords set.
void calculate_total_energy();
//int add_energy(CONNECT *_connect);
int add_forceenergy();
//double bondenergy(BONDLIST &_bondarray);
//double angleenergy(BONDLIST &_anglearray);
//void add_dihedralenergy(BONDLIST &_dihedralarray);
int add_longenergy();
int add_longenergy_with(UNIT *_unit2);
int print_energy();
bool energy_is_ok();
int push_unitlist();
        // Adds unit to the begin of the linked list of all known units:
        // last_in_time->previous_in_time->previous_in_time->...

//========== SIMPLE ACCESSORS ==========//
int get_distance_from_last_cross();
int get_count_of_crosses();
int get_distance_from_chainbegin();
CHAIN *get_chain();
CONNECT *get_connect();
//CONNECT *get_fieldconnect();
//int get_fieldconnectshift();
int set_has_coords(bool _has_coords);
bool get_has_coords();
int get_unitdefid();
UNITID *get_unitid();
int set_unitclass(int _unitclass);
int get_unitclass();

UNITDEF *get_unitdef();
//========== END OF SIMPLE ACCESSORS ==========//

static int dump(const OUTPUTFILENAMES &_filenames);
        // save all units to the given file for later restoration.
int dump_unit(LOGFILE &_dumpfile);
        // save this units to the given file for later restoration.
static int dump_history(LOGFILE &_dumpfile);
int get_unit_realid();
static int restore(string _restorefilename, UNITS *_units, CHAINS *_chains);
        // restore all units from the given file
static int restore_atoms_connections();
static UNIT* get_unit_by_realid(unsigned int _realid);
int set_molecule(MOLECULE *_molecule);
MOLECULE *get_molecule();
static UNIT *find_preset_unit();
static set<ATOM*> list_dropped_atoms();
bool atom_overlaps(int _thisatom_index);
set<ATOM*> list_dropped_atoms(int _thisatom_index);
bool is_referrable(); // true, if this unit can be
		      // referred by its chainid and unitid in a bond.
static int init_units_outputting_order();
static list<UNIT*> get_units_outputting_order(int _molecule);
static int retr_units_outputting_order(int _molecule, list<UNIT*> &_order);
	// if _molecule==-1, then all molecules will be shown.
static int retr_atoms_outputting_order(
		list<UNIT*> &_units, list<ATOM*> &_atoms);
	// _atoms will be list of all outputtable atoms in all units of
	// given list of _units. Side effect: every atom will have its
	// oid set according to its index+1 in the _atoms list.
static int refine_coordinates_from_xyz(const string _xyz_filename);
	// Replace current coordinates of all output-ready atoms
	// with those read from a given xyz-file.
UNIT *get_nextunit();
int distribute_forcelists(CONNECT &_connect, bool _only_internal_bonds);
int distribute_force(BONDCLASS _bondclass, int _numatoms, BOND &_bond, bool _only_internal_bonds);
static int create_all_forcelists();
ATOM * relative_atom(const char * _atomref);
static int replace_all_atoms();
};

#endif
