#include "unit.h"
#include "mcgen.h"
#include "unitdef.h"
#include "readxyz.h"

UNIT *UNIT::last_in_time = 0;
vector<UNIT*> UNIT::inited_units;
list<UNIT*> UNIT::units_outputting_order;
bool UNIT::units_outputting_order_ready=false;
long unsigned int UNIT::non_dumped_counter=0;
long unsigned int UNIT::best_dumped_size=0;
vector<UNIT*> UNIT::paused_units;

//========== PRODUCING OUTPUT ==========

//========== method log_unit_coords ==========

int UNIT::log_unit_coords()
{
int i;
int atomscount=0;
ATOM *a;

logfile.print("### Unit coordinates: ###\n");
for (i=0;i<atoms;i++)
{
	a=&atom[i];
	if (a->get_has_coords())
		logfile.print(
			"%4s %12.5lf %12.5lf %12.5lf # %5d %3d/%04d/%02d\n",
			a->get_label(),
			a->get_xcoord(),
			a->get_ycoord(),
			a->get_zcoord(),
			++atomscount,
			get_chain()->get_chainid(),
			distance_from_chainbegin+1,
			i+1
		);
}
return atomscount;
}

//========== method get_position ==========

const string UNIT::get_position()
{
return unitid.to_str();
}

//========== method print_defname ==========

int UNIT::print_defname()
{
logfile.print(get_defname().c_str());
return 0;
}

//========== method get_defname ==========

string UNIT::get_defname()
{
return unitdef->get_unittype_string();
}

//========== method print_structure_recursive() ==========

int UNIT::print_structure_recursive()
{
print_defname();

if (sideunit)
{
	logfile.print("(");
	sideunit->print_structure_recursive();
	logfile.print(")");
}

if(nextunit)
{
	logfile.print(",");
	nextunit->print_structure_recursive();
}
else
	logfile.print(".");
return 0;
}

//========== method print_structure ==========

int UNIT::print_structure()
{
logfile.print("Structure: ");
print_structure_recursive();
logfile.print("\n\n");
return 0;
}

//========== TAKEBACK ==========

//========== method take_back_until ==========

int UNIT::take_back_until(int _finalsize)
{
//logfile.print("takeback: %d -> %d\n",history.size(),_finalsize);
assert(_finalsize != HISTSIZE_UNKNOWN);
unsigned int finalsize=_finalsize;
assert(history.size()>=finalsize);
while (history.size()>finalsize)
	ATOM::take_back_one_atom();
assert(history.size()==finalsize);
return 0;
}

//========== INITIALISE UNITS ==========

//========== method init ==========

void UNIT::init()
{
chain=0;
unitdef=0;
unitclass=UC_EMPTY;

previousunit=nextunit=sideunit=0;
is_sidechain=false;
distance_from_chainbegin=0;
distance_from_last_cross=0;
count_of_crosses=0;

atoms = 0;
atom = 0;
//is_real = true;
has_coords = false;

// initialize unit's local energy variables
energy.reset();
// initialize total energy variables
energy_cumul.reset();

randstart=params->get_randstart();
totaltried=0;

histsize_before=HISTSIZE_UNKNOWN;
molecule=0;
last_atom=0;
cross=0;
connectfirst=0;
totally_internal=false;
}

//========== constructor UNIT ==========

UNIT::UNIT(UNIT *_previousunit, bool _is_sidechain)
{
int i;

assert (_previousunit!=0);

init();

set_previousunit_relation(_previousunit, _is_sidechain);

chain=previousunit->get_chain();
assert (chain!=0);

molecule=previousunit->molecule;
assert (molecule!=0);
int result;
try
{
	result=choose_unitdef(); // May throw an exception CHAIN_IS_COMPLETED
}
catch(int Exception) // catch(int exception) would not be portable.
{
	if ((! is_sidechain) and (_previousunit->unitclass &
				(UC_REPEAT | UC_CROSS | UC_BEGIN)))
	{
		unitclass=UC_TAIL;
		set_atoms(1);
	}
	else
	{
		unitclass=UC_EMPTY;
		//throw CHAIN_IS_COMPLETED;
	}
	return;
}

if (result==CHAIN_IS_COMPLETED)
{
	if ((! is_sidechain) and (_previousunit->unitclass &
				(UC_REPEAT | UC_CROSS | UC_BEGIN)))
	{
		unitclass=UC_TAIL;
		set_atoms(1);
	}
	else
	{
		unitclass=UC_EMPTY;
		//throw CHAIN_IS_COMPLETED;
	}
	return;
}

set_atoms(unitdef->get_atomnr());
for (i=0;i<atoms;i++)
{
	atom[i].set_atomdef(unitdef->get_atom(i+1));
}
gen_unique_unitid();
if (chain->get_pause())
{
	logfile.print("DEBUG: Unit %s was added to paused_units.\n", unitid.to_c_str());
	paused_units.push_back(this);
}
else
{
	push_unitlist();
}
}

//========== method gen_unique_unitid ==========

int UNIT::gen_unique_unitid()
{
UNITID *previd = previousunit? (&previousunit->unitid) : 0;
unitid.gen_unique(molecule->get_molid(),previd,is_sidechain,
		(distance_from_chainbegin>0),unitdef);
return 0;
}

//========== method choose_unitdef ==========

int UNIT::choose_unitdef()
{
choose_chain();
//logfile.print("Choosing unitdef ... ");

if (chain==0)
{ // chain is completed.
	unitdef = 0;
	unitclass = UC_EMPTY;
        return CHAIN_IS_COMPLETED;
}

unitdef = chain->get_unitdef();
unitclass = unitdef->get_unitclass();

calculate_distances();

bool not_enough_crosses = (count_of_crosses < chain->get_repeat());
if (not_enough_crosses)
{
	int delta=distance_from_last_cross - chain->get_spacing();
	bool should_be_cross = (chain->random_deviation_accept(
				delta, chain->get_tolerance()));
	if (should_be_cross)
	{
		cross=chain->get_next_cross();
		unitdef = cross->crosstype;
		unitclass = UC_CROSS;
		calculate_distances();
	}
}

return 0;
}

//========== method choose_chain ==========

int UNIT::choose_chain()
{
// input: previousunit, is_sidechain
if (is_sidechain) // Sidechain begins
{
	assert(previousunit);
	assert(previousunit->unitdef);
	if (previousunit->unitdef->get_unittype()!=CROSS)
	{
		chain=0;
//              throw CHAIN_IS_COMPLETE;
		return 0;
	}
	assert(previousunit->cross);
	chain=previousunit->cross->sidechain;
	connectfirst=chain->get_random_connectfirst();
}
else if (previousunit != 0) // Main chain continues
{
	int needed_length=previousunit->chain->get_unitnr();
	int has_length=previousunit->distance_from_chainbegin+1;
	bool chain_is_over = previousunit->chain->random_deviation_accept(
			has_length - needed_length, 0);

	if (chain_is_over)
	{ // previousunit was the last one in its chain
		chain = previousunit->chain->get_nextchain();
		connectfirst=chain->get_random_connectfirst();
	} else {
		chain = previousunit->chain;
		connectfirst=0;
	}
}
// else:        This is the first unit.
//              chain has already been set by the UNIT constructor.
return 0;
}

//========== constructor UNIT ==========
// for the virtual seed unit

UNIT::UNIT(CHAIN *_chain)
{
init();
unitclass=UC_SEED;
set_atoms(3);

push_unitlist();

chain=_chain;
}

//========== constructor UNIT ==========
// for units that are given as xyz-files.

UNIT::UNIT(const string _xyz_filename)
{
READXYZ readxyz(_xyz_filename);
init();
unitclass=UC_PRESET;
push_unitlist();

set_atoms(readxyz.size());
for (int i=1;i<=atoms;i++)
{
	ATOM *a=get_atom(i);
	a->set_atomarg(readxyz[i-1].name);
	a->set_coords(&readxyz[i-1].coords);

	ATOMDEF *ad = new ATOMDEF;
	ad->set_method(FIXED);
	ad->set_atomarg(a->get_atomarg());
	ad->set_coords(&readxyz[i-1].coords);
	a->set_atomdef(ad);
}
has_coords = true;
randstart=0;
}

//========== constructor UNIT ==========

UNIT::UNIT()
{
init();
push_unitlist();
}

//========== static method refine_coordinates_from_xyz ==========

int UNIT::refine_coordinates_from_xyz(const string _xyz_filename)
{
READXYZ readxyz(_xyz_filename);

init_units_outputting_order();
list<ATOM*> atoms_order;
retr_atoms_outputting_order(units_outputting_order, atoms_order);

if (readxyz.size() != atoms_order.size()) {
	logfile.print("\n");
	logfile.print("Notice: units_outputting_order.size()=%zd\n",units_outputting_order.size());
	logfile.print("Notice: readxyz.size()=%zd\n",readxyz.size());
	logfile.print("Notice: atoms_order.size()=%zd\n",atoms_order.size());
	assert(readxyz.size()==atoms_order.size());
}

READXYZ::iterator in_it;
list<ATOM*>::iterator out_it;
for (in_it=readxyz.begin(), out_it=atoms_order.begin();
		in_it!=readxyz.begin();
		++in_it, ++out_it)
{
	assert((*out_it)->get_atomarg() == defs->get_atomarg(in_it->name));
	assert((*out_it)->get_has_coords());
	(*out_it)->set_coords(&in_it->coords);
}

recalculate_energy();
return 0;
}

//========== method calculate_distances ==========

int UNIT::calculate_distances()
{
// input: previousunit, is_sidechain, chain, unitdef
// output: distance_from_chainbegin, distance_from_last_cross, count_of_crosses

distance_from_chainbegin = 0;
distance_from_last_cross = 1;
count_of_crosses = 0;

if (not previousunit)
	return 0;
if (not chain)
	return 0;

if ((chain == previousunit->get_chain()) && (not is_sidechain))
{
	distance_from_chainbegin =
		previousunit->get_distance_from_chainbegin()+1;
	distance_from_last_cross =
		previousunit->get_distance_from_last_cross()+1;
	count_of_crosses =
		previousunit->get_count_of_crosses();
}

if (not unitdef)
	return 0;

if (unitdef->get_unittype() == CROSS)
{
	distance_from_last_cross = 0;
	count_of_crosses++;
}

return 0;
}
//========== method set_previousunit_relation ==========

int UNIT::set_previousunit_relation(UNIT *_previousunit, bool _is_sidechain)
{
previousunit=_previousunit;
is_sidechain=_is_sidechain;
if (_previousunit)
{
	if (_is_sidechain)
	       previousunit->sideunit=this;
	else
	       previousunit->nextunit=this;
}
return 0;
}

//========== method recursive_initialization ==========

int UNIT::recursive_initialization()
{
//logfile.print("<rec %p>",this);
if ((not (unitclass & UC_SEED)) and (unitdef == 0))
	return 0;

UNIT *p=this;
for (int i=params->get_nummainfirst();i>0;i--)
{
	//logfile.print("<pre>");
        if (p->nextunit==0)
                p->nextunit=new UNIT(p, false);
        p=p->nextunit;
	if (not p->unitdef)
		break;
}

if (not (unitclass & UC_SEED))
{
	//logfile.print("<side>");
        if (sideunit==0)
                sideunit=new UNIT(this, true);
	if (sideunit->unitdef) // if sidebranch exists
		sideunit->recursive_initialization();
}

//logfile.print("<next>");
if (nextunit==0)
	nextunit=new UNIT(this, false);
nextunit->recursive_initialization();

return 0;
}

//==========  method initialize_paused_units ==========
/*!
    \fn UNIT::initialize_paused_units()
    \brief Repeatedly initializes recursively all the units in paused_list.
 */
int UNIT::initialize_paused_units()
{
for (vector<UNIT*>::iterator it = paused_units.begin();
		it != paused_units.end(); ++it)
{
	UNIT *u=*it;
	u->push_unitlist();
	logfile.print("DEBUG: Unit %s was taken from paused_units.\n", u->unitid.to_c_str());
}
paused_units.clear();
return 0;
}

//========== static method initialize_atoms_placement_order ==========

int UNIT::initialize_atoms_placement_order()
{
for (vector<UNIT*>::iterator it = inited_units.begin();
		it != inited_units.end(); ++it)
{
	UNIT *u=*it;
	for (int i=0;i<u->atoms;++i)
	{
		ATOM *a=&u->atom[i];
		u->last_atom=a->add_to_atoms_placement_order();
	}
}
return 0;
}

//========== static method place_linear ==========

int UNIT::place_linear(MOLECULE *_molecule)
{
//unsigned int current_unitid; // current_unitid = unit's real id
//logfile.print("DEBUG: place_linear %p\n",_molecule);
UNIT *u;
int takeback;
int placeresult;

while (current_unitid<inited_units.size())
{
	if (current_unitid > best_dumped_size)
	{
		best_dumped_size = current_unitid;
		dump(outputnames_best);
	}
	if (non_dumped_counter >= params->get_dump_time())
	{
		non_dumped_counter=0;
		dump(outputnames_regular);
	}
	if (_molecule->if_interrupt())
	{
		_molecule->set_interrupt_done(true);
		dump(outputnames_final);
		exit(INTERRUPT_EXITCODE);
	}
	u=inited_units[current_unitid];
	if (u->molecule != _molecule)
	{
		return 0; // This molecule is complete.
	}
	takeback=0; // number of units to take back
//	printf("histsize: %d\n",history.size());
	if (u->histsize_before == HISTSIZE_UNKNOWN)
		u->histsize_before=history.size();
//	logfile.print("U%d histsize_before=%d\n",u->unit_realid,u->histsize_before);
	placeresult=u->place();
	if ((placeresult == UNIT_PLACE_OK)
		or (placeresult == UNIT_PLACE_EMPTY_UNIT))
	{
		current_unitid++;
		continue;
	}
	if (placeresult == UNIT_PLACE_MAXTRYUNIT_EXCEEDED)
		takeback=TAKEBACK_STEPS_COUNT;
	else if (placeresult == UNIT_PLACE_RANDOM_TAKEBACK)
	{
		if (params->get_takeback_is_fixed())
			takeback=params->get_randval()-1;
		else
			takeback=CHAIN::random_integer(params->get_randval()-1);
		logfile.print("random takeback: %d units. \n",takeback);
	}
	// take back the current unit.
	take_back_until(u->histsize_before);
	u->has_coords=false;
	// take back given number of real units and all virtual units between.
	while ((takeback > 0) and (current_unitid > 0))
	{
		u=inited_units[current_unitid-1];
		if (u->molecule != _molecule)
			break;
		current_unitid--;
		take_back_until(u->histsize_before);
		u->has_coords=false;
		//printf("taking back unit %p\n",u);
		if (u->unitclass & (UC_NORMAL|UC_SEED|UC_TAIL))
			takeback--;
	}
//	printf("take-back until %d\n",u->histsize_before);
//	if (params->dec_triesleft()<=0)
//		return 1;
}
return 0;
}

//========== method place ==========

int UNIT::place()
{
//logfile.print("DEBUG: UNIT %p->place()\n",this);
//logfile.print("Before placing unit there are total of %d atoms placed\n",
//      count_all_placed_atoms());
energy.reset();
if (has_coords)
	return UNIT_PLACE_OK;
if (unitclass & (UC_SEED|UC_TAIL))
{
	while (ATOM::get_placed_count() < last_atom)
	{
		ATOM::place_next(energy);
	}
	//place_seed();
	calculate_total_energy();
	has_coords=true;
	return UNIT_PLACE_OK;
}
if (not (unitclass & UC_NORMAL))
{
	calculate_total_energy();
	has_coords=true;
	return UNIT_PLACE_EMPTY_UNIT;
}

int status;
long int retry;
long int tryno=0;
int unitbegin=history.size();
Failcode failcode;
stat_count_geomfail = 0;
stat_minimal_energy = 999999;

//logfile.print("\ndebug_history_init %d\n",unitbegin);

retry=params->get_maxtryunit();
while (1)
{
	energy.reset();
	status = _ATOM_place_OK_;
	while (ATOM::get_placed_count() < last_atom)
	{
		status=ATOM::place_next(energy);
		if (status == _ATOM_place_GEOMFAIL_)
			break;
	}
	if (status == _ATOM_place_GEOMFAIL_)
		++stat_count_geomfail;
	tryno++;
	params->dec_triesleft(); // Throws exception when reaches to zero.
	totaltried++;
	randstart--;
	retry--;
	non_dumped_counter++;
	molecule->inc_moltries();
	if (status == _ATOM_place_OK_)
	{
		has_coords = true;
		calculate_total_energy();

		if (energy_is_ok())
		{
			failcode = OK;
			logfile.print("%s ",unitid.to_c_str());
			logfile.print(
					"OK %6ld %12d %6ld %12ld %15ld\n",
					tryno, randstart, totaltried,
					molecule->get_moltries(), params->get_triesdone());

			// tryno on katsete arv he uniti kohta ja algvrtustatakse
			// place() funktsiooni igal kivitamisel.
			// 
			// randstart lheb viksemaks ja nitab mitu sammu saab veel teha
			// enne uut juhuslikku tagasivtmist. Kui randstart juab nulli,
			// vetakse juhuslik arv uniteid tagasi.
			//
			// totaltried on uniti parameeter, mis nitab mitu korda on seda
			// unitit proovitud. Algvrtustatakse mcgen kivitamisel.
			// 
			// molecule->get_moltries() on tehtud tryde arv he molekuli kohta.
			// 
			// get_triesdone on katsete arv kigi unitite kohta kokku alates
			// mcgen kivitamisest.

			print_energy();
			return UNIT_PLACE_OK;
		}
		else
		{
			failcode = ENERGYFAIL;
		}
	}
	else
	{
		failcode = GEOMFAIL;
	}

//      logfile.print("failed.\n");

	if ((randstart <= 0) or (retry<=0))
	{
		has_coords=false;
		take_back_until(unitbegin);
		logfile.print("%s ",unitid.to_c_str());
		logfile.print(
				"NO %6ld %12d %6ld %12ld %15ld\n",
				tryno, randstart, totaltried,
				molecule->get_moltries(), params->get_triesdone());
		if (verbose_stat)
		{
			if (failcode == ENERGYFAIL)
			{
				print_energy();
			}
			logfile.print("Statistics: geometry failed %d times, minimum energy was %lf\n",
					stat_count_geomfail, stat_minimal_energy);
		}
		if (randstart <= 0)
		{
			randstart=params->get_randstart();
			return UNIT_PLACE_RANDOM_TAKEBACK;
		}
		else // if (retry <= 0)
		{
			return UNIT_PLACE_MAXTRYUNIT_EXCEEDED;
		}
	}

	has_coords=false;
	take_back_until(unitbegin);
}
return UNIT_PLACE_OK;
}

//========== method place_seed ==========

int UNIT::place_seed()
{
int atomid;
ATOM *thisatom;

assert(!has_coords);

for (atomid=1;atomid<=atoms;atomid++)
{
	thisatom=get_atom(atomid);
	if (! thisatom->get_has_coords())
		thisatom->place_randomly();
}
calculate_total_energy();
has_coords=true;
return UNIT_PLACE_OK;
}

//========== static method write_coords ==========

int UNIT::write_coords(const OUTPUTFILENAMES &_filenames)
{
//logfile.print("DEBUG: write_coords\n");
UNIT *u;
double totalenergy=0;
GENXYZ genxyz;

set<ATOM*> dropped_atoms=list_dropped_atoms();

// Cycle through all initialized units
init_units_outputting_order();
for (list<UNIT*>::iterator it =units_outputting_order.begin();
			     it!=units_outputting_order.end(); ++it)
{
	u=*it;
	if (u->unitclass & (UC_PRESET|UC_INVISIBLE))
	{
		//logfile.print("DEBUG: Writing UNIT %p coordinates\n",&u);
		for (int i=0;i<u->atoms;i++)
		{
			ATOM &a=u->atom[i];
			if (dropped_atoms.count(&a) == 0)
			{
				//logfile.print("DEBUG: Writing ATOM %p coordinates\n",&u->atom[i]);
				genxyz.push_back(&a);
			}
		}
	}
	else if ((u->unitclass & UC_NORMAL) && (u->has_coords))
	{
		//logfile.print("DEBUG: Writing UNIT %p coordinates\n",&u);
		totalenergy+=u->energy.total();
		for (int i=0;i<u->atoms;i++)
		{
			//logfile.print("DEBUG: Writing ATOM %p coordinates\n",&u->atom[i]);
			if (not u->atom[i].get_has_coords())
				assert(false);
			genxyz.push_back(&u->atom[i]);
		}
	}
}
logfile.print("(Total energy: %lf)",totalenergy);
genxyz.write_history("mcgen.history");
genxyz.write_xyz(_filenames.xyz);
GENXYZ::write_listed_coords(_filenames.dropxyz,dropped_atoms);

genfieldoutini->set_filename(_filenames.fieldout);
genfieldoutini->set_field_filename(_filenames.dlpolyfield);
genfieldoutini->write();

return 0;
}

//========== ENERGY CHECK ==========

//========== method fieldenergy_is_computable ==========

bool UNIT::fieldenergy_is_computable(CONNECT *_connect, bool _verbose)
{
return bondenergy_is_computable(*_connect->get_bondarray(_BOND_) ,2,_verbose)
and bondenergy_is_computable(*_connect->get_bondarray(_ANGLE_)   ,3,_verbose)
and bondenergy_is_computable(*_connect->get_bondarray(_DIHEDRAL_),4,_verbose);
}
	
//========== method bondenergy_is_computable ==========

bool UNIT::bondenergy_is_computable(BONDLIST &_bondarray, uint _bondatomsnum,
		bool _verbose)
{
unsigned int k;
BONDLIST::iterator iter;
for (iter=_bondarray.begin(); iter != _bondarray.end(); ++iter)
{
        BOND &b=*iter;
	for (k=1; k<=_bondatomsnum; k++)
	{
		UNIT *u=relative_unit(b.get_unitXid(k));
		if ((u->chain->get_chainid()!=b.get_chainXid(k))
		and (u->chain->get_chainid()!=0))
		{
			if (_verbose)
				logfile.print("Bad chain identifier given: chainid %d (must be %d) unitid %d atomid %d at field file [%d].\n",
					b.get_chainXid(k),
					u->chain->get_chainid(),
					b.get_unitXid(k),
					b.get_atomXid(k),
					b.get_fieldlinenr());
			return false;
		}
		//try{relative_atom(b,k);} catch(string _msg){return false;}
	}
}
return true;
}

//========== method calculate_energy ==========
//
//double UNIT::calculate_energy()
//{
//energy.reset();
//
//add_forceenergy();
//add_longenergy();
//calculate_total_energy();
//return energy.total();
//}

//========== method calculate_total_energy ==========

void UNIT::calculate_total_energy()
{
UNIT *u=previous_in_time;
if (u)
{
	energy_cumul=energy+u->energy;
} else {
	energy_cumul=energy;
}
}

//========== static method recalculate_energy ==========

int UNIT::recalculate_energy()
{
int all_previous_units_have_coords=1;
for (vector<UNIT*>::iterator it = inited_units.begin();
		it != inited_units.end(); ++it)
{
	all_previous_units_have_coords &= (*it)->has_coords;
	if ((*it)->has_coords)
	{
		assert(all_previous_units_have_coords);
		(*it)->energy.reset();
		// TODO make it working with energy in individual atoms
		//(*it)->calculate_energy();
	}
}
return 0;
}

//========== method longenergy ==========

int UNIT::add_longenergy()
{
UNIT *u;
assert(has_coords);
for (u=last_in_time;u!=0;u=u->previous_in_time)
        if (u->has_coords and (u->unitclass & (UC_NORMAL | UC_PRESET)))
		// UC_INVISIBLE puhul energiat ei arvutata
	{
		add_longenergy_with(u);
	}
return 0;
}

//========== method add_longenergy_with ==========

int UNIT::add_longenergy_with(UNIT *_unit2)
{
int i, j, j_limit;
ATOM *thisatom, *secondatom;

if (((_unit2 == this) or (_unit2 == previousunit))
		and (unitclass & UC_BEGIN))
        return 0;

// _unit2==this should not be needed, but it is for some reason.
if (((_unit2 == this) or (_unit2 == previousunit)) 
		and (previousunit->unitclass & UC_BEGIN))
        return 0;

for (i=1;i<=atoms;i++)
{
        thisatom=get_atom(i);
        if (_unit2==this)
                j_limit = i-1;
        else
                j_limit = _unit2->atoms;
        for (j=1;j<=j_limit;j++)
        {
                secondatom=_unit2->get_atom(j);
                if ((not thisatom->has_connection_with(secondatom))
	       	and (not thisatom->get_atomarg()->is_invisible_for(secondatom->get_atomarg()))
	       	and (not secondatom->get_atomarg()->is_invisible_for(thisatom->get_atomarg())))
                {
			double distance=ATOM::distance(thisatom, secondatom);
                        energy.vdwen+=ATOM::vdw_energy
				(thisatom, secondatom, distance);
                        energy.couen+=ATOM::coulomb_energy
				(thisatom, secondatom, distance);
                }
        }
}
return 0;
}

//========== method print_energy ==========

int UNIT::print_energy()
{
logfile.print(
"      Bond energy increase: %12.4lf,       total bond energy: %12.4lf\n",
energy.bonen, energy_cumul.bonen);
logfile.print(
"     Angle energy increase: %12.4lf,      total angle energy: %12.4lf\n",
energy.angen, energy_cumul.angen);
logfile.print(
"  Dihedral energy increase: %12.4lf,   total dihedral energy: %12.4lf\n",
energy.dihen, energy_cumul.dihen);
if (verbose_energy) logfile.print(
"       Vdw energy increase: %12.4lf\n   Coulomb energy increase: %12.4lf\n",
energy.vdwen, energy.couen);
logfile.print(
"Long range energy increase: %12.4lf, total long range energy: %12.4lf\n",
energy.lonen(), energy_cumul.lonen());
logfile.print(
"     Total energy increase: %12.4lf,      Grand total energy: %12.4lf\n",
energy.total(),energy_cumul.total());
return 0;
}

//========== method energy_is_ok ==========

bool UNIT::energy_is_ok()
{
bool result;
double unit_energy=energy.total();

if (unit_energy < stat_minimal_energy)
	stat_minimal_energy=unit_energy;

if ((unitclass & UC_END)
		and (chain->get_surefinish() > 0)
		and (totaltried > chain->get_surefinish()))
	return true;
	// This endunit has been tried quite a number of times already,
	// let's just accept it at last despite its too large energy.
	
if (unit_energy<0)
        result=true; // Energy IS ok.
else
{       // roll a dice on whether to accept that energy or not.
        double rand=randome->RANDOME_output_random_number();
        result=rand<ATOM::energy_exp(unit_energy);
        if (unitdef->get_unittype()==BEGIN)
                result=true;    // No energy check for beginunits.
}
//logfile.print("\ndebug_energy %lf %d START\n",unit_energy,result);
//log_unit_coords();
//print_energy();
//logfile.print("\ndebug_energy %lf %d END\n",unit_energy,result);
//if (!result)
//	logfile.print("<!energy>");
return result;
}

//========== ACCESSORS AND MEMORY MANAGEMENT ==========

//========== method get_atom ==========

ATOM *UNIT::get_atom(int _atomid)
{
//logfile.print(" get_atom(%d) ",_atomid);
if ((_atomid<1) or (_atomid>atoms))
{
      ostringstream message;
      message<<"Bad atom index: "<<_atomid<<
              " not in range [1,"<<atoms<<"] for unit "<<
	      get_defname()<<". Current unit is "<<
	      inited_units[current_unitid]->get_defname();
      throw message.str();
}
return &atom[_atomid-1];
}

//========== method set_atoms ==========

int UNIT::set_atoms(int _atoms)
{
SYSUTILS tmp;

if (atoms==0)
{
        atoms=_atoms;
        atom=new ATOM[atoms];
        tmp.allocate_error(atom,"UNIT","ATOMs array",_SYSUTILS_MEMERR);
	for (int i=0;i<atoms;i++)
	{
		ATOM &a=atom[i];
		a.set_unit(this);
		a.set_index(i+1);
	}
}
else
        assert(atoms==_atoms);
return 0;
}

//========== method get_atoms ==========

int UNIT::get_atoms()
{
return atoms;
}

//========== method is_referrable ==========

bool UNIT::is_referrable()
{
if (! this)
	return false;
if (unitclass & (UC_NORMAL | UC_SEED))
	return true;
return false;
}

//========== method find_unit_from_tree ==========

UNIT *UNIT::find_unit_from_tree(int _chainid, int _unitid)
{
bool wasside[LONGEST_BACKREF+1];
UNIT *ups[LONGEST_BACKREF+1];
int curshift = 0;
ups[0]=this;
bool backwards = true;

assert(_unitid>=-LONGEST_BACKREF);
assert(_unitid<=0);

while(true)
{
	// move straight until the required distance is approached.
	while(curshift < -_unitid)
	{
		wasside[curshift]=false;
		if (backwards)
			ups[curshift+1]=ups[curshift]->previousunit;
		else
			ups[curshift+1]=ups[curshift]->nextunit;
		if (ups[curshift+1]->chain != chain)
			totally_internal = false;
		if (ups[curshift+1]->unitdef != unitdef)
			totally_internal = false;
		curshift++;
	}
	backwards=false;

	// if chainid does match, call it a success
	int chainid=ups[curshift]->chain->get_chainid();
	if ((chainid==_chainid) or (chainid==0))
		return ups[curshift];
	
	// else take back until last straight move and replace that with
	// a sidechain move.
	while(curshift>0)
	{
		curshift--;
		if ((not wasside[curshift]) and (ups[curshift]->sideunit != 0)
		and (not (ups[curshift]->sideunit->unitclass & UC_EMPTY)))
		{
			wasside[curshift]=true;
			ups[curshift+1]=ups[curshift]->sideunit;
			if (ups[curshift+1]->chain != chain)
				totally_internal = false;
			if (ups[curshift+1]->unitdef != unitdef)
				totally_internal = false;
			curshift++;
			break;
		}
	}

	// No success at all.
	ostringstream msg;
	msg<<"Cannot find unit with unitid "<<_unitid<<" that has chainid "
		<<_chainid<<".";
	throw msg.str();
}
}

//========== method relative_atom ==========

ATOM *UNIT::relative_atom ( int _chainid, int _unitid, int _atomid, bool _only_internal_bonds )
{
	bool ups[LONGEST_BACKREF+1];
	ups[0]=this;

	UNIT *u;
	if ( ( _unitid<0 ) and not ( _only_internal_bonds ) )
		u=find_unit_from_tree ( _chainid, _unitid );
	else
	{
		u=relative_unit ( _unitid );
		if ( !u )
			totally_internal = false;
		else if ( u->chain->get_chainid() != _chainid )
			totally_internal = false;
		else if ( u->unitdef != unitdef )
			totally_internal = false;

		if (!u)
			return 0;
		
		if ( not _only_internal_bonds )
		{
			int chainid=u->chain->get_chainid();
				
			if ( ( chainid!=_chainid ) and ( chainid!=0 ) )
			{
				ostringstream message;
				message<<"Wrong chain identifier given: chainid "<<_chainid
				<<" unitid "<<_unitid<<" atomid "<<_atomid
				<<". Chainid must be "<<u->chain->get_chainid()
				<<" here.";
				throw message.str();
			}
		}
	}
	ATOM *a=u->get_atom ( _atomid );
	return a;
}

//========== method relative_atom ==========

ATOM *UNIT::relative_atom(FOUR_ATOMREF &_atomref, int _index, bool _only_internal_bonds)
{
return relative_atom(_atomref.get_chainXid(_index),
                        _atomref.get_unitXid(_index),
                        _atomref.get_atomXid(_index),
			_only_internal_bonds);
}

//========== method push_unitlist ==========

int UNIT::push_unitlist()
{
// Add current unit to the lists of all initialized units
inited_units.push_back(this);
unit_realid=inited_units.size();

previous_in_time=last_in_time;
last_in_time=this;
units_outputting_order_ready=false;
return 0;
}

//========== SIMPLE ACCESSORS ==========//

//========== method get_distance_from_last_cross ==========

inline int UNIT::get_distance_from_last_cross()
{
return distance_from_last_cross;
}

//========== method get_count_of_crosses ==========

inline int UNIT::get_count_of_crosses()
{
return count_of_crosses;
}

//========== method get_distance_from_chainbegin ==========

inline int UNIT::get_distance_from_chainbegin()
{
return distance_from_chainbegin;
}

//========== method get_chain ==========

inline CHAIN *UNIT::get_chain()
{
return chain;
}

//========== method set_has_coords ==========

int UNIT::set_has_coords(bool _has_coords)
{
has_coords=_has_coords;
return 0;
}

//========== method get_has_coords ==========

bool UNIT::get_has_coords()
{
return has_coords;
}

//========== method get_unitdefid ==========

inline int UNIT::get_unitdefid()
{
//return unitdefid;
return unitdef->get_unitid();
}

//========== method get_unitid ==========

UNITID *UNIT::get_unitid()
{
return &unitid;
}

//========== method get_unitdef ==========

inline UNITDEF *UNIT::get_unitdef()
{
if (this)
        return unitdef;
else
        return 0;
}

//========== method set_unitclass ==========

int UNIT::set_unitclass(int _unitclass)
{
unitclass=_unitclass;
return 0;
}

//========== method get_unitclass ==========

int UNIT::get_unitclass()
{
return unitclass;
}

//========== END OF SIMPLE ACCESSORS ==========//

//========== method get_unit_realid ==========

int UNIT::get_unit_realid()
{
if (this)
        return unit_realid;
else
        return 0;
}

//========== method dump ==========

int UNIT::dump(const OUTPUTFILENAMES &_filenames)
{
logfile.print("<dump>");
{
        // Create dump file
        LOGFILE dumpfile("mcgen.dump.new");
        dumpfile.set_verbose(false);

        dumpfile.print("CURRENT %d %d\n",current_molid,current_unitid);
	
        dumpfile.print("# UNITS %ld\n",(long)inited_units.size());

        // Cycle through all initialized units
        for (unsigned int i=0;i<inited_units.size();i++)
                inited_units[i]->dump_unit(dumpfile);

        dumpfile.print("END_OF_UNITS \n");
        dumpfile.print("RANDOM %ld %ld\n",
                randome->RANDOME_output_seed1(),
                randome->RANDOME_output_seed2());

	dump_history(dumpfile);

	dumpfile.print("TRIESDONE %ld\n",params->get_triesdone());

	for (int i=1; i<=chains->get_molnr(); ++i)
	{
		MOLECULE *mol=chains->get_molecule(i);
		dumpfile.print("MOLTRIES %ld %s # %d\n",
				mol->get_moltries(),
				mol->get_interrupt_done()?"YES":"NO",
				i);
	}
	
//      dumpfile.print("# non_dumped_counter %lu\n",    non_dumped_counter);
//      non_dumped_counter is alway zero when doing a dump.
//      So it is not necessary to save it.
        dumpfile.print("CLOSE \n");
        // dumpfile is closed here automatically by the end of the scope.
}
// Write over old dumpfile
rename("mcgen.dump.new",_filenames.dump.c_str());
write_coords(_filenames);
logfile.print("\n");
return 0;
}

//========== method dump_unit ==========

int UNIT::dump_unit(LOGFILE &_dumpfile)
{
// start of unit record
_dumpfile.print("UNIT %d ",unit_realid);

// dump unitdef. Obsolete, because unitdef is obtained from CHAIN.
if (unitdef)
        _dumpfile.print("%s %d\n",
                unitdef->get_unittype_str().c_str(),
                unitdef->get_unitid());
else
        _dumpfile.print("none \n");

_dumpfile.print("UNITCLASS %d\n",unitclass);
_dumpfile.print("MOLECULE %d\n",molecule->get_molid());
		
// dump pointer to previous unit.
_dumpfile.print(
	"PREVIOUSUNIT %d%s # NEXTUNIT %d SIDEUNIT %d PREVIOUS_IN_TIME %d\n",
        previousunit->get_unit_realid(),
        is_sidechain?" SIDE":"",
        nextunit->get_unit_realid(),
        sideunit->get_unit_realid(),
	previous_in_time->get_unit_realid());

_dumpfile.print("CHAIN %d\n",chain->get_chainid());

_dumpfile.print(
	"# dist_last_cross %d distance_chainbegin %d, count_crosses %d\n",
        distance_from_last_cross, distance_from_chainbegin, count_of_crosses);
_dumpfile.print(
	"FLAGS %s %s %d %ld\n",
        has_coords?"YES":"NO", (unitclass&UC_REAL)?"YES":"NO", randstart, totaltried);
//_dumpfile.print(
//	"UNITENERGY %0.15lf %0.15lf %0.15lf %0.15lf %0.15lf\n",
//        unit_energy, bonen, angen, dihen, lonen);
//_dumpfile.print(
//	"TOTALENERGY %0.15lf %0.15lf %0.15lf %0.15lf %0.15lf\n",
//        uenergy_tot, benergy_tot, aenergy_tot, denergy_tot, lenergy_tot);
// _dumpfile.print("CONNECT %d\n", geomconnect->get_connect_id());

_dumpfile.print("HISTSIZE_BEFORE %d\n",histsize_before);
// dump all atoms coordinates in the unit.
_dumpfile.print("ATOMS %d\n",atoms);
for (int j=1;j<=atoms;j++)
{
        ATOM *a=get_atom(j);
	bool atom_needs_coords = a->get_has_coords();
	if ((unitclass & UC_NORMAL) and has_coords)
		atom_needs_coords=false;
	if (unitclass & (UC_PRESET | UC_INVISIBLE))
		atom_needs_coords=false;
	_dumpfile.print("ATOM %d %s %s",
			j,
			a->get_has_coords()?"YES":"NO",
			atom_needs_coords?"YES":"NO");
	if (atom_needs_coords)
		_dumpfile.print(" %12.6lf %12.6lf %12.6lf",
				a->get_xcoord(),
				a->get_ycoord(),
				a->get_zcoord());
	_dumpfile.print("\n");
}
return 0;
}

//========== static method dump_history ==========
int UNIT::dump_history(LOGFILE &_dumpfile)
{
stack<ATOM*> reverse;
unsigned int histsize=history.size();

_dumpfile.print("HISTORY %d\n",histsize);
while (not history.empty())
{
	reverse.push(history.top());
	history.pop();
}
while (not reverse.empty())
{
	ATOM *a=reverse.top();
	history.push(a);
	_dumpfile.print("ATOM %d %d\n",
			a->get_unit()->get_unit_realid(),
			a->get_index());
	reverse.pop();
}

assert(history.size()==histsize);
return 0;
}

//========== static method restore ==========
// restores all units from the given file

int UNIT::restore(string _restorefilename, UNITS *_units, CHAINS *_chains)
{
TEXTFILE resto(_restorefilename);

// Read: CURRENT current_molid current_unitid
resto.read_next("CURRENT");
current_molid=resto.read_lineint(2);
current_unitid=resto.read_lineint(3);

while (true)
{
        resto.read_next();
	// Read: UNIT unitid
        if (not resto.SYSU_test_string(1,"UNIT"))
                break;
	int read_unitrealid=resto.read_lineint(2);
        // logfile.print("[unit %d]", read_unitrealid);
        UNIT *u = get_unit_by_realid(read_unitrealid);
	if (! u)
		u = new UNIT();
        //resto.read_next("UNIT");
        //logfile.print("Restoring unit %d as %d\n",resto.read_lineint(2),u->unit_realid);
	assert(u->unit_realid==read_unitrealid);
	Unittype ut=UNITDEF::str2unittype(resto.SYSU_read_linestring(3));
	if (ut != INVALID)
		u->unitdef=_units->get_unitdef(ut,resto.read_lineint(4));
	else
		u->unitdef=0;

	// Read: UNITCLASS unitclass
	resto.read_next("UNITCLASS");
	u->set_unitclass(resto.read_lineint(2));
	
	// Read: MOLECULE molecule
	resto.read_next("MOLECULE");
	u->set_molecule(_chains->get_molecule(resto.read_lineint(2)));

	if (u->unitclass & UC_SEED)
		u->get_molecule()->set_seedunit(u);
	
	// Read: PREVIOUSUNIT prev
	resto.read_next("PREVIOUSUNIT");
	int prev=resto.read_lineint(2);
	bool is_side=resto.SYSU_test_string(3,"SIDE");
	u->set_previousunit_relation(get_unit_by_realid(prev),is_side);

	// Read: CHAIN chainid
        // restore chain
        resto.read_next("CHAIN");
        u->chain=u->get_molecule()->get_chain(resto.read_lineint(2));
	// // restore unitdef
        // u->unitdef=u->chain->get_unitdef();

	
        // restore flags
	
	// NoRead: dist_last_cross, ...

	// Read: FLAGS flags
        resto.read_next("FLAGS");
        u->has_coords=resto.read_linebool(2);
//        u->is_real=resto.read_linebool(3);
        u->randstart=resto.read_lineint(4);
        u->totaltried=resto.read_linelong(5);
        // restore distances
        u->calculate_distances();

	/*
	// Read: UNITENERGY unit_energy bonen angen dihen lonen
	resto.read_next("UNITENERGY");
	u->unit_energy=resto.read_linedouble(2);
	u->bonen=resto.read_linedouble(3);
	u->angen=resto.read_linedouble(4);
	u->dihen=resto.read_linedouble(5);
	u->lonen=resto.read_linedouble(6);

	// Read: TOTALENERGY uenergy_tot benergy_tot aenergy_tot denergy_tot lenergy_tot
	resto.read_next("TOTALENERGY");
	u->uenergy_tot=resto.read_linedouble(2);
	u->benergy_tot=resto.read_linedouble(3);
	u->aenergy_tot=resto.read_linedouble(4);
	u->denergy_tot=resto.read_linedouble(5);
	u->lenergy_tot=resto.read_linedouble(6);
	*/

	// Read: CONNECT connect
	resto.read_next("CONNECT");
//	u->geomconnect=CONNECT::find_connect(resto.read_lineint(2));
//	u->fieldconnect=u->geomconnect;

	// Read: HISTSIZE_BEFORE histsize_before
	resto.read_next("HISTSIZE_BEFORE");
	u->histsize_before=resto.read_lineint(2);
        
        resto.read_next("ATOMS");
        u->set_atoms(resto.read_lineint(2));
        for (int atomid=1;atomid<=u->atoms;atomid++)
        {
                resto.read_next("ATOM");
                int atomid_in=resto.read_lineint(2);
                if (atomid_in != atomid)
                {
                        logfile.print(
			"Wrong atomid in restorefile: is %d, must be %d\n",
			atomid_in,atomid);
			throw("Incorrect restorefile.");
                }
		if (u->get_unitdef() != 0)
			u->get_atom(atomid)->set_atomdef(
				u->get_unitdef()->get_atom(atomid));
                bool hascoords_in=resto.read_linebool(3);
		if (u->unitclass & (UC_PRESET | UC_INVISIBLE))
			assert(hascoords_in);
                bool hascoords_dumped=resto.read_linebool(4);
                if (hascoords_dumped)
                {
                        double x_in=resto.read_linedouble(5);
                        double y_in=resto.read_linedouble(6);
                        double z_in=resto.read_linedouble(7);
			if (u->unitclass & (UC_PRESET|UC_INVISIBLE))
				assert(u->get_atom(atomid)->
					check_coords(x_in, y_in, z_in));
			else
	                        u->get_atom(atomid)->
					set_coords(x_in, y_in, z_in);
                }
		//else assert(hascoords_in == 
		//		u->get_atom(atomid)->get_has_coords());
        }
	if (u->unitclass & UC_NORMAL)
		u->gen_unique_unitid();
}
assert(resto.SYSU_test_string(1,"END_OF_UNITS"));
// restore randome seed
resto.read_next("RANDOM");
randome->RANDOME_input_seed(
        resto.read_linelong(2),
        resto.read_linelong(3));

// restore history
resto.read_next("HISTORY");
unsigned int histsize=resto.read_lineint(2);
assert(history.empty());
for (unsigned int i=0;i<histsize;i++)
{
	resto.read_next("ATOM");
	UNIT *u=get_unit_by_realid(resto.read_lineint(2));
	ATOM *a=u->get_atom(resto.read_lineint(3));
	history.push(a);
}
assert(history.size()==histsize);

resto.read_next("TRIESDONE");
params->set_triesdone(resto.read_linelong(2));

for (int i=1; i<=chains->get_molnr(); ++i)
{
	MOLECULE *mol=chains->get_molecule(i);
	resto.read_next("MOLTRIES");
	mol->set_moltries(resto.read_linelong(2));
	mol->set_interrupt_done(resto.read_linebool(3));
}
	
// TODO for each molecule
//resto.read_next("INTERRUPT_AT_TOTALTRIES");
//molecule::set_interrupt_at_totaltries(resto.read_linelong(2));

//logfile.print("[end]");
resto.read_next("CLOSE");
//choose_fieldconnects(0);
restore_atoms_connections();
return 0;
}

//========== method restore_atoms_connections ==========

int UNIT::restore_atoms_connections()
{
stack<ATOM*> reverse;
unsigned int histsize=history.size();

while (not history.empty())
{
	reverse.push(history.top());
	history.pop();
}
while (not reverse.empty())
{
	ATOM *a=reverse.top();
	history.push(a);
	a->restore_connected_atoms();
	reverse.pop();
}
assert(history.size()==histsize);
return 0;
}

//========== method get_unit_by_realid ==========

UNIT* UNIT::get_unit_by_realid(unsigned int _realid)
{
if ((_realid >= 1) and (_realid <= inited_units.size()))
        return inited_units[_realid-1];
else
        return 0;
}

//========== method relative_unit ==========

UNIT *UNIT::relative_unit(int _unitid)
{
UNIT *p;
p=this;
assert (_unitid<=11);
if (_unitid==1)
{
        return nextunit;
}
else if (_unitid==2)
{
        return nextunit->nextunit;
}
else if (_unitid==3)
{
        return nextunit->nextunit->nextunit;
}
else if (_unitid==4)
{
        return nextunit->nextunit->nextunit->nextunit;
}
else if (_unitid==5)
{
        return nextunit->nextunit->nextunit->nextunit->nextunit;
}
else if (_unitid==10)
{
	if (sideunit->chain != chain)
		totally_internal = false;
	if (sideunit->unitdef != unitdef)
		totally_internal = false;
        return sideunit;
}
else
{
        while (_unitid++ < 0)
	{
                p=p->previousunit;
		if (!p)
			totally_internal = false;
		else if (p->chain != chain)
			totally_internal = false;
		else if (p->unitdef != unitdef)
			totally_internal = false;
	}
        return p;
}
}

/*//========== method relative_unit ==========

UNIT *UNIT::relative_unit(char *_unitid)
{
UNIT *p;
p=this;

while (*_unitid != 0)
{
	switch (*_unitid)
	{
		'p': p=this->previousunit; break;
		'n': p=this->nextunit; break;
		's': p=this->sideunit; break;
		'0'-'9': break;
		default: assert(false);
	}
	++_unitid;
}
return p;
}*/

//========== method set_molecule ==========

int UNIT::set_molecule(MOLECULE *_molecule)
{
	molecule=_molecule;
	return 0;
}

//========== method get_molecule ==========

MOLECULE *UNIT::get_molecule()
{
//	if (! molecule)
//		throw "Molecule of an unit without molecule was asked for.";
	return molecule;
}

//========== static method find_preset_unit ==========

UNIT *UNIT::find_preset_unit()
{
UNIT *preset_unit=0;
// Cycle through all initialized units
vector<UNIT*>::iterator u,end;
end=inited_units.end();
for(u=inited_units.begin(),end=inited_units.end();u<end;u++)
{
        if ((*u)->unitclass & UC_PRESET)
	{
		assert(preset_unit==0);
		preset_unit=*u;
	}
}
return preset_unit;
}

//========== static method list_dropped_atoms ==========

set<ATOM*> UNIT::list_dropped_atoms()
{
set<ATOM*> result;

UNIT *preset_unit=find_preset_unit();
if (! preset_unit)
	return result;

assert(result.empty());
for (int j=1;j<=preset_unit->atoms;j++)
{
	if (preset_unit->atom_overlaps(j))
	{
		set<ATOM*> partial_result = preset_unit->
			list_dropped_atoms(j);
		// result := union (result, partial_result)
		copy(partial_result.begin(), partial_result.end(),
				inserter(result,result.begin()));
	}
}
return result;
}

//========== method atom_overlaps ==========

bool UNIT::atom_overlaps(int _thisatom_index)
{
ATOM *thisatom=get_atom(_thisatom_index);

// Cycle through all initialized units
vector<UNIT*>::iterator u=inited_units.begin();
vector<UNIT*>::iterator end=inited_units.end();
for(;u<end;u++)
{
	if ((*u)==this)
		continue;
	if (not ((*u)->unitclass & (UC_NORMAL | UC_PRESET)))
		continue;
	for (int k=1;k<=(*u)->atoms;k++)
	{
		ATOM *otheratom=(*u)->get_atom(k);
		if (thisatom->overlaps_with(*otheratom))
		{
			return true;
		}
	}
}
return false;
}

//========== method list_dropped_atoms ==========

set<ATOM*> UNIT::list_dropped_atoms(int _thisatom_index)
{
set<ATOM*> result;

assert(unitclass & (UC_PRESET|UC_INVISIBLE));

ATOM *thisatom=get_atom(_thisatom_index);
result.insert(thisatom);
//logfile.print("<DROP: %d>\n",_thisatom_index);
if (! fieldin)
	return result;

int mol_atom_shift=0;
int mol_max_atom=0;
for (int m=0; m<fieldin->FI_retr_molecules(); m++)
{
	mol_atom_shift=mol_max_atom;
	mol_max_atom+=fieldin->FI_retr_nummols(m) * fieldin->FI_retr_atoms(m);
	if (mol_max_atom >= _thisatom_index)
	{
		int atom_id_in_mol = (_thisatom_index - mol_atom_shift - 1) %
			(fieldin->FI_retr_atoms(m)) + 1;
		//logfile.print("<find: %d,%d>\n",m,atom_id_in_mol);
		vector<int> rigid = fieldin->FI_find_rigid_sites(m,
				atom_id_in_mol);
		for (uint s=0; s<rigid.size(); s++)
		{
		//	logfile.print("<got: %d>\n",rigid[s]);
			int dropid=rigid[s]+_thisatom_index-atom_id_in_mol;
		//	logfile.print("<DROP other atom in RIGID block: %d>\n",
		//	dropid);
			result.insert(get_atom(dropid));
		}
		break;
	}
}
return result;
}

//========== method init_units_outputting_order ==========

int UNIT::init_units_outputting_order()
{
if (units_outputting_order_ready)
	return 0;
units_outputting_order.clear();
vector<UNIT*>::iterator u,end;
//logfile.print("DEBUG: units_outputting_order follows\n");
for(u=inited_units.begin(),end=inited_units.end();u<end;++u)
	if ((*u)->unitclass & (UC_PRESET|UC_INVISIBLE))
	{
		assert((*u)->molecule==0);
		assert((*u)->has_coords);
		units_outputting_order.push_back(*u);
		//logfile.print("DEBUG: unit %p\n",*u);
	}
for (int molid=1;molid<=chains->get_molnr();molid++)
{
	//logfile.print("DEBUG: units in molecule %d\n",molid);
	init_units_outputting_order_recursion(
			chains->get_molecule(molid)->get_seedunit());
}
units_outputting_order_ready=true;
return 0;
}

//========== method init_units_outputting_order_recursion ==========

int UNIT::init_units_outputting_order_recursion(UNIT *_where)
{
for (UNIT *u=_where;u;u=u->nextunit)
{
	units_outputting_order.push_back(u);
	if (u->sideunit)
		init_units_outputting_order_recursion(u->sideunit);
}
return 0;
}

//========== method get_units_outputting_order ==========

list<UNIT*> UNIT::get_units_outputting_order(int _molecule)
{
int previous_unit_mol_id=0;
list<UNIT*> temp;
for (list<UNIT*>::iterator it = units_outputting_order.begin();
		it!=units_outputting_order.end(); ++it)
{
	UNIT *unit=*it;
	int mol_id=unit->get_molecule()->get_molid();
	assert (previous_unit_mol_id <= mol_id);
	previous_unit_mol_id = mol_id;
	if ((mol_id == _molecule) && (unit->unitclass & UC_REAL))
		temp.push_back(unit);
}
return temp;
}

//========== method retr_units_outputting_order ==========

int UNIT::retr_units_outputting_order(int _molecule, list<UNIT*> &_order)
{
_order.clear();
int previous_unit_mol_id=0;
for (list<UNIT*>::iterator it = units_outputting_order.begin();
		it!=units_outputting_order.end(); ++it)
{
	UNIT *unit=*it;
	int mol_id=unit->get_molecule()->get_molid();
	assert (previous_unit_mol_id <= mol_id);
	previous_unit_mol_id = mol_id;
	if (((mol_id == _molecule) || (_molecule==-1)) &&
			(unit->unitclass & UC_REAL))
		_order.push_back(unit);
}
return 0;
}

//========== method retr_atoms_outputting_order ==========

int UNIT::retr_atoms_outputting_order(list<UNIT*> &_units, list<ATOM*> &_atoms)
{
_atoms.clear();
int previous_unit_mol_id=0;
int oid=0;
for (list<UNIT*>::iterator it = _units.begin(); it!=_units.end(); ++it)
{
	UNIT *unit=*it;
	int mol_id=unit->get_molecule()->get_molid();
	assert (previous_unit_mol_id <= mol_id);
	previous_unit_mol_id = mol_id;

	for (int aid=1; aid<=unit->get_atoms(); ++aid)
	{
		ATOM *atom=unit->get_atom(aid);
		if (atom->get_has_coords())
		{
			_atoms.push_back(atom);
			atom->set_oid(++oid);
		}
	}

}
return 0;
}

//========== method get_nextunit ==========

UNIT *UNIT::get_nextunit()
{
return nextunit;
}

//========== method get_previousunit ==========

UNIT *UNIT::get_previousunit() const
{
return previousunit;
}



/*!
    \fn UNIT::distribute_forcelists(CONNECT &_connect, bool _only_internal_bonds)
 */
int UNIT::distribute_forcelists ( CONNECT &_connect, bool _only_internal_bonds )
{
	assert ( &_connect );

	// Add bonds
	BONDLIST *_bondarray=_connect.get_bondarray ( _BOND_ );
	assert ( _bondarray );
	BONDLIST::iterator it, end;
	end=_bondarray->end();
	for ( it=_bondarray->begin(); it != end; ++it )
	{
		distribute_force ( _BOND_, 2, *it, _only_internal_bonds );
	}

	// Add angles
	BONDLIST *_anglearray=_connect.get_bondarray ( _ANGLE_ );
	assert ( _anglearray );
	end=_anglearray->end();
	for ( it=_anglearray->begin(); it != end; ++it )
	{
		distribute_force ( _ANGLE_, 3, *it, _only_internal_bonds );
	}

	// Add dihedrals
	BONDLIST *_dihedralarray=_connect.get_bondarray ( _DIHEDRAL_ );
	assert ( _dihedralarray );
	end=_dihedralarray->end();
	for ( it=_dihedralarray->begin(); it != end; ++it )
	{
		distribute_force ( _DIHEDRAL_, 4, *it, _only_internal_bonds );
	}

	return 0;
}



/*!
    \fn UNIT::distribute_force(BONDCLASS _bondclass, int _numatoms, BOND &_bond)
    \brief Converts a BOND into BONDINST by replacing relative atom references with absolute atom pointers. Thereafter adds it to a atom's forcelist.
    \param _bondclass shows whether it is EXCLUDE, BOND, ANGLE or DIHEDRAL
    \param _numatoms  is the number of atoms in this bond: 2, 3 or 4.
    \param _bond      is the bond to be converted
    \param _only_internal_bonds if true, all bonds between different CHAINs will be ignored.
 */
int UNIT::distribute_force(BONDCLASS _bondclass, int _numatoms, BOND &_bond, bool _only_internal_bonds)
{
	ATOM *a[4];
	for (int i=0; i<4; ++i)
		a[i]=0;
	totally_internal = true;
	for (int i=0; i<_numatoms; ++i)
	{
		a[i]=relative_atom ( _bond, i+1, _only_internal_bonds );
		if ((_only_internal_bonds) && (!a[i]))
			return 0;
		assert(a[i]);
	}
	int row=_bond.get_fieldlinenr();

	//logfile.print("Note: working on field[%d], _only_internal_bonds=%d\n",row, _only_internal_bonds);

	BONDINST bi=BONDINST ( _bondclass, _bond.get_bonddef(),
	                       a[0], a[1], a[2], a[3], row );
	ATOM *l= ( ATOM* ) bi.latest_atom();

	if (totally_internal or not _only_internal_bonds)
	{
		if (_bondclass != _EXCLUDE_)
		{
			l->add_to_forcelist ( bi );
		}
		bi.add_to_connection_lists();
	}
	return 0;
}

//========== static method create_all_forcelists ==========

int UNIT::create_all_forcelists()
{
// Cycle through all initialized units
vector<UNIT*>::iterator it,end;
end=inited_units.end();
for(it=inited_units.begin();it!=end;++it)
{
	UNIT &u = **it;

	if (! (u.chain))
		continue;

	if ( u.unitclass & UC_CROSS )
	{
		CONNECT &cc=*(u.cross->connectcross);
		u.distribute_forcelists(cc, false);
	}
	else
	{
		CONNECT &ic=*(u.chain->get_inconnect());
		u.distribute_forcelists(ic, true);
	}
	
	if ( u.connectfirst )
	{
		assert(u.previousunit);
		assert(u.previousunit->get_chain() != u.get_chain());
		CONNECT &cf=*u.connectfirst;
		u.distribute_forcelists(cf, false);
	}
}
return 0;
}

/*!
    \fn UNIT::relative_atom(const char * _atomref)
 */
ATOM * UNIT::relative_atom(const char * _atomref)
{
UNIT *p = this;
int a=0;

char *it = (char*)_atomref;
while (*it != 0)
{
	switch (*it)
	{
		case('p'): p=p->previousunit; break;
		case('n'): p=p->nextunit; break;
		case('s'): p=p->sideunit; break;
		case('0'):
		case('1'):
		case('2'):
		case('3'):
		case('4'):
		case('5'):
		case('6'):
		case('7'):
		case('8'):
		case('9'): a=(10*a)+((*it)-'0'); break;
		default: assert(false);
	}
	if (p->chain != chain)
		totally_internal = false;
	if (p->unitdef != unitdef)
		totally_internal = false;
	++it;
}
return p->get_atom(a);
}


/*!
    \fn UNIT::replace_all_atoms()
 */
int UNIT::replace_all_atoms()
{
// Cycle through all initialized units
vector<UNIT*>::iterator it,end;
end=inited_units.end();
for(it=inited_units.begin();it!=end;++it)
{
	UNIT *u = *it;

	if ( u->unitclass & UC_CROSS )
	{
		CONNECT &cc=*(u->cross->connectcross);
		cc.do_replace_atoms(u);
	}
	else
	{
		CONNECT &ic=*(u->chain->get_inconnect());
		ic.do_replace_atoms(u);
	}
	
	if ( u->connectfirst )
	{
		assert(u->previousunit);
		assert(u->previousunit->get_chain() != u->get_chain());
		CONNECT &cf=*u->connectfirst;
		cf.do_replace_atoms(u);
	}
}
return 0;
}
