// Functions:	distance
//		invert
//		image
//		inbox
//
// Last changed: Wed May 10 10:41:20 CEST 2000
//
// Description:
//
// 'distance' calculates the vector length
// 'invert' inverts the 3-dim matrix
// 'image' calculates the minimum image of the atom pairs within the
// specified cell
// 'inbox' packs the atoms into the specified cell
//	'direction_shift' shifts a coordinate
//
// Include files:
//      none
//
// Used external function [name, location, header file]:
//	none
//
//========== Parameters description ==========
//
//========== Function distance ==========
//
// INPUT parameters
//
// int nc		number of atoms
// double *x		difference of x coordinates of two atoms
// double *y		difference of y coordinates of two atoms
// double *z		difference of z coordinates of two atoms
//
// OUTPUT parameters
//
// double *dist		calculated vector length
//
//========== Function invert ==========
//
// INPUT parameters
//
// double *a		matrix
// double d		determinant
//
// OUTPUT parameters
//
// double *b		inverse matrix
//
//========== Function image ==========
//
// INPUT parameters
//
// int imcon		image convention
// int idnode		node number
// int mxnode		number of nodes
// int natms		number of atoms
// double *cell		cell parameters
// double *disx		difference of the x coorinates of two atoms
// double *disy         difference of the y coorinates of two atoms
// double *disz         difference of the z coorinates of two atoms
//
// OUTPUT parameters
//
// double *disx         minimum difference of the x coorinates of two atoms
// double *disy         minimum difference of the y coorinates of two atoms
// double *disz         minimum difference of the z coorinates of two atoms
//
//========== Function inbox ==========
//
// INPUT parameters
//
// int imcon		image convention
// int natms		number of atoms
// double *cell		cell parameters
// double *x		x coordinate
// double *y		y coordinate
// double *z		z coordinate
//
// OUTPUT parameters
//
// double *x		new x coordinate
// double *y		new y coordinate
// double *z		new z coordinate
//
//========== Function direction_shift ==========
//
// INPUT parameters
//
// double origcoord		original coordinate
// double dcell				unit cell value of the direction
// double dshift			additional shift
//
// OUTPUT parameters
//
// double direction_shift		shifted coordinate
//
//========== Description of imcon ==========
//
//     for
//     imcon=0 no boundary conditions apply   
//     imcon=1 standard cubic boundaries apply
//     imcon=2 orthorhombic boundaries apply  
//     imcon=3 parallelepiped boundaries apply
//     imcon=4 truncated octahedron boundaries apply
//     imcon=5 rhombic dodecahedron boundaries apply
//     imcon=6 x-y parallelogram boundary conditions : no periodicity in z
//     
//     note: in all cases the centre of the cell is at (0,0,0)
//     warning - replicated data version: does not re-merge
//     coordinate arrays
//
//========== End of the parameters description ==========
//==========================
//========== Code ==========
//==========================

#include <stdio.h>
#include <math.h>

//=======================================
//========== Function distance ==========
//=======================================

void distance (int nc,double *dist, double *x, double *y, double *z)
{
	int i;
	for(i=0;i<nc;i++)
	{
		dist[i]=sqrt(x[i]*x[i]+y[i]*y[i]+z[i]*z[i]);
	}
	return;
}

//=====================================
//========== Function invert ==========
//=====================================

void invert(double *a,double *b,double d)
{

//=======================================================================
//     
//     dl_poly subroutine to invert a 3 * 3 matrix using cofactors
//     
//     copyright - daresbury laboratory 1992
//     author    - w. smith       april 1992
//     
//     wl
//     1996/02/15 14:32:58
//     1.1.1.1
//     Exp
//     
//=======================================================================
     
      
//      real*8 a,b,d,r

//      double a[9],b[9];
	double r;

	  b[0]=a[4]*a[8]-a[5]*a[7];
      b[1]=a[2]*a[7]-a[1]*a[8];
      b[2]=a[1]*a[5]-a[2]*a[4];
      b[3]=a[5]*a[6]-a[3]*a[8];
      b[4]=a[0]*a[8]-a[2]*a[6];
      b[5]=a[2]*a[3]-a[0]*a[5];
      b[6]=a[3]*a[7]-a[4]*a[6];
      b[7]=a[1]*a[6]-a[0]*a[7];
      b[8]=a[0]*a[4]-a[1]*a[3];
//c     
//c     calculate determinant
      d=a[0]*b[0]+a[3]*b[1]+a[6]*b[2];
      r=0.;
      if(fabs(d)>0.)r=1./d;
//c     
//c     complete inverse matrix
      b[0]=r*b[0];
      b[1]=r*b[1];
      b[2]=r*b[2];
      b[3]=r*b[3];
      b[4]=r*b[4];
      b[5]=r*b[5];
      b[6]=r*b[6];
      b[7]=r*b[7];
      b[8]=r*b[8];
	  return;
}

//====================================
//========== Function image ==========
//====================================

void image(int imcon,int idnode,int mxnode,int natms,double *cell,
           double *disx,double *disy,double *disz)
{
     
//=======================================================================
//     
//     dl_poly subroutine for calculating the minimum image
//     of atom pairs within a specified MD cell
//     
//     parallel replicated data version
//     
//     copyright - daresbury laboratory 1992
//     author    - w. smith march 1992.
//     T3D optimised version. t.forester july 1994
//     
//     for
//     imcon=0 no boundary conditions apply
//     imcon=1 standard cubic boundaries apply
//     imcon=2 orthorhombic boundaries apply
//     imcon=3 parallelepiped boundaries apply
//     imcon=4 truncated octahedron boundaries apply
//     imcon=5 rhombic dodecahedron boundaries apply
//     imcon=6 x-y parallelogram boundary conditions : no periodicity in z
//     
//     note: in all cases the centre of the cell is at (0,0,0)
//     warning - replicated data version: does not re-merge 
//     coordinate arrays
//     
//     wl
//     1996/02/15 14:32:57
//     1.1.1.1
//     Exp
//     
//=======================================================================

      double rcell[9],aaa,bbb,ccc,det,xss,yss,zss,ssx,ssy,ssz;
	  int iatm1,iatm2,i;

      if(imcon>0)
	  {

//c     
//c/     block indices

        iatm1 = (idnode*natms)/mxnode;//+1;
        iatm2 = ((idnode+1)*natms)/mxnode-1;

      }
      
      if(imcon==1)
	  {
//c     
//c     standard cubic boundary conditions
        
        
        aaa=1./cell[0];


        for(i=iatm1;i<=iatm2;i++)
		{
          disx[i]=disx[i]-cell[0]*rint(aaa*disx[i]);
          disy[i]=disy[i]-cell[0]*rint(aaa*disy[i]);
          disz[i]=disz[i]-cell[0]*rint(aaa*disz[i]);
		}
	  } 
      else if(imcon==2)
	  {
//c     
//c     rectangular (slab) boundary conditions
        
        aaa=1./cell[0];
        bbb=1./cell[4];
        ccc=1./cell[8];

        for(i=iatm1;i<=iatm2;i++)
		{
          
          disx[i]=disx[i]-cell[0]*rint(aaa*disx[i]);
          disy[i]=disy[i]-cell[4]*rint(bbb*disy[i]);
          disz[i]=disz[i]-cell[8]*rint(ccc*disz[i]);
		}
	  }
      else if(imcon==3)
	  {
//c     
//c     parallelepiped boundary conditions
        
        invert(cell,rcell,det);
        
        for(i=iatm1;i<=iatm2;i++)
		{
          
          ssx=(rcell[0]*disx[i]+rcell[3]*disy[i]+rcell[6]*disz[i]);
          ssy=(rcell[1]*disx[i]+rcell[4]*disy[i]+rcell[7]*disz[i]);
          ssz=(rcell[2]*disx[i]+rcell[5]*disy[i]+rcell[8]*disz[i]);
          
          xss=ssx-rint(ssx);
          yss=ssy-rint(ssy);
          zss=ssz-rint(ssz);
          
          disx[i]=(cell[0]*xss+cell[3]*yss+cell[6]*zss);
          disy[i]=(cell[1]*xss+cell[4]*yss+cell[7]*zss);
          disz[i]=(cell[2]*xss+cell[5]*yss+cell[8]*zss);
		}
	  }
      else if(imcon==4)
	  {
//c     
//c     truncated octahedral boundary conditions
        
        if(!(fabs(cell[0]-cell[4])<1.e-6&&
			fabs(cell[4]-cell[8])<1.e-6))
		{
			puts("error 130");
			return;
		}
        
        aaa=1./cell[0];
        
        for(i=iatm1;i<=iatm2;i++)
		{
          disx[i]=disx[i]-cell[0]*rint(aaa*disx[i]);
          disy[i]=disy[i]-cell[0]*rint(aaa*disy[i]);
          disz[i]=disz[i]-cell[0]*rint(aaa*disz[i]);
          
          if((fabs(disx[i])+fabs(disy[i])+fabs(disz[i]))>=
           (0.75*cell[0]))
		  {
            
            disx[i]=disx[i]-0.5*copysign(cell[0],disx[i]);
            disy[i]=disy[i]-0.5*copysign(cell[0],disy[i]);
            disz[i]=disz[i]-0.5*copysign(cell[0],disz[i]);
            
          }
          
        }
	  }
	return;

}

//====================================
//========== Function inbox ==========
//====================================

int inbox(int imcon,int natms,double *cell,double *x,double *y,double *z)
{
	int i;
	if(imcon!=2 &&imcon!=1)
		return 1;
	for(i=0;i<natms;i++)
	{
		if(z[i]<0)
			z[i]+=(fabs(rint(z[i]/cell[8]))+1)*cell[8];
		if(z[i]>cell[8])
			z[i]-=(rint(z[i]/cell[8]))*cell[8];

		if(x[i]<0)
			x[i]+=(fabs(rint(x[i]/cell[0]))+1)*cell[0];
		if(x[i]>cell[0])
			x[i]-=(rint(x[i]/cell[0]))*cell[0];
		if(y[i]<0)
			y[i]+=(fabs(rint(y[i]/cell[4]))+1)*cell[4];
		if(y[i]>cell[4])
			y[i]-=(rint(y[i]/cell[4]))*cell[4];
	}
	return 0;
}

//==============================================
//========== Function direction_shift ==========
//==============================================

double direction_shift(double origcoord,double dcell,double dshift)
{
double ocoord;
ocoord=origcoord;
if(ocoord<0.0) ocoord=ocoord+dcell;

return ocoord;
};


//=====================================
//========== End of the code ==========
//=====================================
