// $Id: chains.cpp,v 1.49 2009/07/07 08:48:57 endel Exp $

#include "chains.h"
#include <sstream>
#include <string>

//========== constructor CHAINS ==========

CHAINS::CHAINS ( const string _inputfilename, const string _fieldfilename, UNITS &_units )
{
	try
	{
		inputFileName=_inputfilename;
		file=sysu.open_file ( inputFileName,"r" );

		fieldFileName=_fieldfilename;
		fieldfile=fieldsysu.open_file ( fieldFileName,"r" );

		read_next ( "STRUCTURE" );
		int structure=sysu.read_lineint ( 2 ); // how many diferent molecules are in the file.
		int molid=0;

		sysu.nextfilerecord ( file );
		for ( int structid=1; structid<=structure; structid++ )
		{
			MOLECULE mol;

			int multiplenr=read_molecule ( &mol, _units );
			if ( multiplenr<1 )
				throw CHAINS_ERROR ( "Chain must have multiplenr >= 1." );

			// Save a number of duplicates of mol into the molecules array.
			for ( int i=1;i<=multiplenr;i++ )
			{
				// Read geomconstraints for every molecule instance
				// (if available)
				if ( sysu.SYSU_test_string ( 1,"GEOMCONSTRNR" ) )
				{
					mol.set_geomconstrnr ( sysu.read_lineint ( 2 ) );
					sysu.nextfilerecord ( file );
					for ( int k=1;k<=mol.get_geomconstrnr();k++ )
					{
						sysu.test_filelinestringlnr ( 1,string ( "GEOM" ),
						                              "IOERROR:",inputFileName,_SYSUTILS_IOERR );
						GEOMCONSTR *g=mol.get_geomconstr ( k );
						g->set_gmtype ( sysu.read_lineint ( 2 ) );
						for ( int j=1;j<=7;j++ )
							g->set_gmpar ( j,sysu.read_linedouble ( 2+j ) );
						sysu.nextfilerecord ( file );
					}
					if ( sysu.SYSU_test_string ( 1,"INTERRUPT" ) )
					{
						mol.set_interrupt_number_of_tries ( sysu.read_linelong ( 2 ) );
						mol.set_interrupt_filled_per_cent ( sysu.read_linedouble ( 3 ) );
						sysu.nextfilerecord ( file );
					}
				} // Else this molecule has the same geometric constraints
				// as the previous one.
				mol.set_molid ( ++molid );
				molecules.push_back ( new MOLECULE ( mol ) );
			}
		}

		string readword = sysu.SYSU_read_linestring ( 1 );
		chomp ( readword );
		if ( readword != "CLOSE" )
			throw CHAINS_ERROR ( "CLOSE expected." );
	
		sysu.close_file ( file );

		read_nextfield ( "CLOSE" );
		fieldsysu.close_file ( fieldfile );

//for (uint k=0;k<molecules.size();k++)
//	printf("molecule[%d] id=%d\n",k,molecules[k]->get_molid());
	}
	catch ( CHAINS_ERROR &e )
	{
		ostringstream msg;
		msg<<inputFileName<<":"<<sysu.SYSU_retr_linecount() <<": ";
		msg<<e<<"\n";
		logfile.print ( msg.str().c_str() );
		exit ( 1 );
	}
	catch ( FIELD_ERROR &e )
	{
		ostringstream msg;
		msg<<fieldFileName<<":"<<fieldsysu.SYSU_retr_linecount() <<": ";
		msg<<e<<"\n";
		logfile.print ( msg.str().c_str() );
		exit ( 1 );
	}
}

//========== method read_molecule ==========

int CHAINS::read_molecule ( MOLECULE *_mol, UNITS &_units )
{
	int multiplenr;

	sysu.test_filelinestringlnr ( 1,string ( "CHAINS" ),
	                              "IOERROR:",inputFileName,_SYSUTILS_IOERR );
		multiplenr=sysu.read_lineint ( 2 ); // how many copies of this molecule will be.

		read_nextfield ( "FIELDS" );

// Initialize chain[0] as seed chain for the virtual seedunit.
		_mol->new_chain();
		assert ( _mol->get_chainnr() == 0 );
		CHAIN *seedchain=_mol->get_chain ( 0 );
		seedchain->set_chainid ( 0 );
		seedchain->set_unitdef ( 0 );
		seedchain->set_unitnr ( 1 );

		sysu.nextfilerecord ( file ); //read_next ( "CHAIN" );
		CHAIN *nextchain = read_chains ( _mol, _units );
		seedchain->set_nextchain ( nextchain );

	return multiplenr;
}

//========== method read_connect ==========

int CHAINS::read_connect ( CONNECT *_connect )
{
	try
	{
		/*		_connect->set_defs(defs);
				_connect->set_constid ( sysu.SYSU_read_linestring ( ++k ) );
				_connect->set_cangid ( sysu.SYSU_read_linestring ( ++k ) );
				_connect->set_cdihid ( sysu.SYSU_read_linestring ( ++k ) );
				for ( int i=1;i<=3;i++ )
				{
					_connect->set_chainXid ( i, sysu.read_lineint ( ++k ) );
					_connect->set_unitXid ( i, sysu.read_lineint ( ++k ) );
					_connect->set_atomXid ( i, sysu.read_lineint ( ++k ) );
				}*/

		while ( sysu.nextfilerecord ( file ) == 0 )
		{
			string codeword = sysu.SYSU_read_linestring ( 1 );
			chomp ( codeword );
			if ( codeword == "REPLACE" )
			{
				short int k=1;
				REPLACE_ATOM r;
				r.set_replace_ref ( sysu.SYSU_read_linestring ( ++k ) );
				r.set_defs ( defs );
				string atomtype=sysu.SYSU_read_linestring ( ++k );
				ATOMARG *atomarg = defs->get_atomarg ( atomtype );
				if (atomarg == 0)
					throw (string("Unknown atom \"")+atomtype+"\"");
				r.set_atomarg ( atomarg );
				// Only available genmeth here is RANDOM.
				r.set_constid ( sysu.SYSU_read_linestring ( ++k ) );
				r.set_cangid ( sysu.SYSU_read_linestring ( ++k ) );
				r.set_cdihid ( sysu.SYSU_read_linestring ( ++k ) );
				r.set_method ( sysu.SYSU_read_linestring ( ++k ).c_str() );
				r.set_chainXid( 1,sysu.read_lineint ( ++k ) );
				r.set_unitXid ( 1,sysu.read_lineint ( ++k ) );
				r.set_atomXid ( 1,sysu.read_lineint ( ++k ) );
				r.set_chainXid( 2,sysu.read_lineint ( ++k ) );
				r.set_unitXid ( 2,sysu.read_lineint ( ++k ) );
				r.set_atomXid ( 2,sysu.read_lineint ( ++k ) );
				r.set_chainXid( 3,sysu.read_lineint ( ++k ) );
				r.set_unitXid ( 3,sysu.read_lineint ( ++k ) );
				r.set_atomXid ( 3,sysu.read_lineint ( ++k ) );
				r.set_has_coords ( false );
				_connect->add_replace_atom ( r );
			}
			else if ( codeword == "END_CONNECT" )
			{
				return 0;
			}
			else
			{
				throw string("REPLACE or END_CONNECT expected");
			}
		}
		throw string("File ended before END_CONNECT");
	}
	catch ( string &e )
	{
		ostringstream msg;
		msg<<inputFileName<<":"<<sysu.SYSU_retr_linecount() <<": ";
		msg<<e<<"\n";
		logfile.print ( msg.str().c_str() );
		exit ( 1 );
	}
	return 0;
}

//========== method read_next ==========

int CHAINS::read_next ( const char *_codeword )
{
	string tmpString=_codeword;

	sysu.nextfilerecord ( file );
	string readword = sysu.SYSU_read_linestring ( 1 );
	chomp ( readword );
	if ( readword != tmpString )
		throw CHAINS_ERROR(tmpString + " expected.");
	return 0;
}

//========== method read_lineunittype ==========

Unittype CHAINS::read_lineunittype ( int _position )
{
	char tmp[_SYSUTILS_INPROW_LEN];

	sysu.read_linestring ( tmp, _position );

	if ( !strcmp ( tmp,"beginunit" ) )
		return BEGIN;
	else if ( !strcmp ( tmp,"repeatunit" ) )
		return REPEAT;
	else if ( !strcmp ( tmp,"crosstunit" ) )
		return CROSS;
	else if ( !strcmp ( tmp,"endunit" ) )
		return END;
	else
	{
		logfile.print ( "Error: unknown unittype '%s'.\n", tmp );
		logfile.print ( "'beginunit', 'repeatunit', 'crossunit' or 'endunit' expected.\n" );
		throw(CHAINS_ERROR("'beginunit', 'repeatunit', 'crossunit' or 'endunit' expected.\n"));
		exit ( 1 );
	}
}

//========== method read_nextfield ==========

int CHAINS::read_nextfield ( const char *_codeword )
{
	string tmpString=_codeword;
	return read_nextfield ( tmpString );

/*
	fieldsysu.nextfilerecord ( fieldfile );
	string readword = fieldsysu.SYSU_read_linestring ( 1 );
	chomp ( readword );
	if ( readword != tmpString )
		throw FIELD_ERROR(tmpString + " expected.");
	return 0;


//logfile.print("Read next field line: %s\n",_codeword);
	fieldsysu.nextfilerecord ( fieldfile );
	if ( not fieldsysu.SYSU_test_string ( 1,tmpString ) )
	{
		string line=fieldsysu.SYSU_read_line100char();
		chomp(line);
		{
			ostringstream msg;
			//msg<<"IOERROR: '"<<tmpString<<"' expected at "<<
			//	get_fieldfilemarker()<<
			//	", but '"<<line<<"' found instead.";
			msg<<tmpString<<"' expected, but '"<<line<<"' found instead.";
			throw FIELD_ERROR ( msg.str() );
		}
	}
//test_filelinestring(1,tmpString,"IOERROR:",fieldFileName,_SYSUTILS_IOERR);
	return 0;

*/
}

//========== method read_nextfield ==========

int CHAINS::read_nextfield ( const string &_codeword )
{
	fieldsysu.nextfilerecord ( fieldfile );
	string readword = fieldsysu.SYSU_read_linestring ( 1 );
	chomp ( readword );
	if ( readword != _codeword )
		throw FIELD_ERROR(_codeword + " expected.");
	return 0;
}

//========== method read_bond ==========

int CHAINS::read_bond ( const char *_codeword,
                        BONDCLASS _bondclass, int _nrofatoms,
                        int _bondid, BOND &_bond )
{
	int /*id,*/ i, k;
	try
	{
		assert ( _nrofatoms>=2 );
		assert ( _nrofatoms<=4 );

		read_nextfield ( _codeword );
		_bond.set_fieldlinenr ( fieldsysu.SYSU_retr_linecount() );
		k=2;
		/*id=fieldsysu.read_lineint(k++);
		if (id != _bondid)
		{
			logfile.print("Error: %s id %d found where %d expected in %s.\n",
					_codeword,id,_bondid,fieldFileName.c_str());
			exit(1);
		}*/
		if ( _bondclass != _EXCLUDE_ )
			_bond.set_bondid ( _nrofatoms, fieldsysu.SYSU_read_linestring ( k++ ) );
		for ( i=1;i<=_nrofatoms;i++ )
		{
			_bond.set_chainXid ( i, fieldsysu.read_lineint ( k++ ) );
			_bond.set_unitXid ( i, fieldsysu.read_lineint ( k++ ) );
			_bond.set_atomXid ( i, fieldsysu.read_lineint ( k++ ) );
		}
		if ( _bondclass == _EXCLUDE_ )
			_bond.sort();
	}
	catch ( string Exception )
	{
		throw FIELD_ERROR ( Exception );
	}
	return 0;
}

//========== method read_bonds ==========

int CHAINS::read_bonds ( BONDLIST *_bondarray, const string _targetstr,
                         int _atomnr, BONDCLASS _bondclass, const char* _bondclassstr )
{
	unsigned int id;

	read_nextfield ( _targetstr );
// TODO check unitid or crossid
	assert ( _bondarray->size() ==0 );
	_bondarray->resize ( fieldsysu.read_lineint ( 2 ) );
	BONDLIST::iterator iter, lopp;
	lopp=_bondarray->end();
	for ( iter=_bondarray->begin(); iter != lopp; ++iter )
		read_bond ( _bondclassstr,_bondclass, _atomnr,id,*iter );
	return 0;
}

//========== method read_bondgroup ==========

int CHAINS::read_bondgroup ( FIELDTARGET _target, string _targetstr,
                             CONNECT *_connect )
{
	read_bonds ( _connect->get_bondarray ( _EXCLUDE_ ),
	             _targetstr+"EXCLUDE", 2, _EXCLUDE_, "EXC" );
	read_bonds ( _connect->get_bondarray ( _BOND_ ),
	             _targetstr+"BONDS", 2, _BOND_, "BOND" );
	read_bonds ( _connect->get_bondarray ( _ANGLE_ ),
	             _targetstr+"ANGLES", 3, _ANGLE_, "ANGLE" );
	read_bonds ( _connect->get_bondarray ( _DIHEDRAL_ ),
	             _targetstr+"DIHEDRALS", 4, _DIHEDRAL_, "DIH" );
	return 0;
}

//========== method get_molecule ==========

MOLECULE *CHAINS::get_molecule ( unsigned int _molid )
{
	if ( _molid==0 )
		return 0;
	if ( ( _molid<1 ) or ( _molid>molecules.size() ) )
	{
		ostringstream message;
		message<<"Bad molecule identificator: "<< _molid
		<<" not in range [1,"<<molecules.size() <<"]";
		throw CHAINS_ERROR ( message.str() );
	}
	return molecules[_molid-1];
}

//========== method get_molnr ==========

int CHAINS::get_molnr()
{
	return molecules.size();
}

//========== method get_chainsfilemarker ==========

string CHAINS::get_chainsfilemarker()
{
	ostringstream msg;
	msg<<inputFileName<<":"<<sysu.SYSU_retr_linecount() <<":";
	return msg.str();
}

//========== method get_fieldfilemarker ==========

string CHAINS::get_fieldfilemarker()
{
	ostringstream msg;
	msg<<fieldFileName<<"["<<fieldsysu.SYSU_retr_linecount() <<"]";
	return msg.str();
}


// ========= Exception constructors

CHAINS_ERROR::CHAINS_ERROR ( const char* _msg ) : string ( _msg )
{};

CHAINS_ERROR::CHAINS_ERROR ( const string _msg ) : string ( _msg )
{};

FIELD_ERROR::FIELD_ERROR ( const char* _msg ) : string ( _msg )
{}

FIELD_ERROR::FIELD_ERROR ( const string _msg ) : string ( _msg )
{}



/*!
    \fn CHAINS::chain_description_is_complete()
    \brief Checks whether all important parts of a chain are given. Executed after the last line of a chain is parsed.
 */
bool CHAINS::check_chain_description_completeness ( CHAIN *_chain )
{
	if ( ( _chain->get_chainid() !=1 ) and ( _chain->get_cfnr() == 0 ) )
		throw CHAINS_ERROR ( "Chain with chainid > 1 must have at least one CONNECTFIRST record." );
	return true;
}


/*!
    \fn CHAINS::read_chains(MOLECULE *_molecule)
    \brief Reads all main chains or one side chain in a molecule from the CHAINS and FIELD files. Returns the chainid of the first chain read.
 */
CHAIN *CHAINS::read_chains ( MOLECULE *_mol, UNITS &_units )
{
	CHAIN *thischain = 0;
	CHAIN *first_chain = 0;
	int chainid = 0;
	Unittype unittype = INVALID;
	bool pauseflag = false;	// If this is true, then all units until the end of the side chain
				// (until exiting this subroutine) will be marked as paused.
	do
	{
		string codeword = sysu.SYSU_read_linestring ( 1 );
		chomp(codeword);

		if ( codeword == "CHAIN" )
		{
			if ( thischain )
				check_chain_description_completeness ( thischain );
			int prev_chainid = chainid;
			chainid = _mol->new_chain();
			thischain=_mol->get_chain ( chainid );
			if ( prev_chainid != 0 )
				_mol->get_chain ( prev_chainid ) -> set_nextchain ( thischain );

			if ( first_chain == 0 )
				first_chain = thischain;

			int id=sysu.read_lineint ( 2 );
			if ( id != chainid )
			{
				ostringstream msg;
				msg<<"Chainid is "<<id<<", must be "<<chainid<<".";
				throw CHAINS_ERROR ( msg.str() );
			}

			thischain->set_unitnr ( sysu.read_lineint ( 3 ) );
			unittype=read_lineunittype ( 4 );
			int unitid=sysu.read_lineint ( 5 );
			UNITDEF *unitdef=_units.get_unitdef ( unittype, unitid );
			thischain->set_unitdef ( unitdef );
			if ( unittype == END )
				thischain->set_surefinish (
				    sysu.read_lineint ( 6 ) );

			// Check for abnormal chain combinations.
			if ( ( chainid==1 ) and ( unittype != BEGIN ) )
				throw CHAINS_ERROR ( "Chain with chainid=1 must consist of BEGINUNITs." );
			if ( ( chainid!=1 ) and ( unittype == BEGIN ) )
				throw CHAINS_ERROR ( "Chain with chainid > 1 must not consist of BEGINUNITs." );

			thischain->set_pause(pauseflag);
			logfile.print("DEBUG: CHAIN on line %d has pauseflag value %d.\n", sysu.SYSU_retr_linecount(), pauseflag);

			read_nextfield ( "FIELD" );
			fieldsysu.SYSU_error_exit_if_not_true (
			    fieldsysu.read_lineint ( 2 ) ==thischain->get_chainid(),
			    "mcgen","fieldid does not match chainid.",_SYSUTILS_IOERR );

			read_bondgroup ( _IN_, "IN", thischain->get_inconnect() );
		}
		else if ( codeword == "CROSSES" )
		{
			if ( chainid == 0 )
				throw CHAINS_ERROR ( "\"CHAIN\" expected before \"CROSSES\"." );

			if ( unittype != REPEAT )
				throw CHAINS_ERROR ( "\"CROSSES\" not expected for that unit type." );

			thischain->set_repeat ( sysu.read_lineint ( 2 ) );
			string randword = sysu.SYSU_read_linestring ( 3 );
			chomp(randword);
			if ( randword == "RANDOM" )
				thischain->set_cross_random ( true );
			else if ( randword == "SEQUENTIAL" )
				thischain->set_cross_random ( false );
			else throw CHAINS_ERROR (
				    string ( "Cross choosing method is \"" ) +randword+
				    "\", must be \"RANDOM\" or \"SEQUENTIAL\"." );
		}
		else if ( codeword == "CROSS" )
		{
			if ( chainid == 0 )
				throw CHAINS_ERROR ( "\"CHAIN\" or \"END_CROSS\" expected before \"CROSS\"." );

			if ( unittype != REPEAT )
				throw CHAINS_ERROR ( "\"CROSS\" not expected." );

			CROSS_RECORD thiscross;

			thiscross.crosstype = _units.get_unitdef ( CROSS,sysu.read_lineint ( 2 ) );

			thiscross.spacing = sysu.read_lineint ( 3 );
			if ( thiscross.spacing <= params->get_nummainfirst() )
			{
				ostringstream str;
				str<<"CROSSUNIT spacing ("<<thiscross.spacing
				<<") must be greater than the number of\n"
				<<"pre-generated main chain units ("
				<<params->get_nummainfirst() <<").\n"
				<<"Error at chain "<<chainid<<".\n";
				throw CHAINS_ERROR ( str.str() );
			}

			thiscross.tolerance = sysu.read_lineint ( 4 );

			thiscross.connectcross = new CONNECT();

			read_next ( "CONNECTCROSS" );
			read_connect ( thiscross.connectcross );
			read_bondgroup ( _CC_, "CC", thiscross.connectcross );

			//read_next ( "CHAIN" );
			sysu.nextfilerecord ( file );

			CHAIN *sidechain = read_chains ( _mol, _units );
			thiscross.sidechain = sidechain;

			string cw = sysu.SYSU_read_linestring ( 1 );
			chomp(cw);

			if ( cw != "END_CROSS" )
				throw CHAINS_ERROR ( "\"END_CROSS\" expected." );

			thischain->new_cross ( thiscross );
		}
		else if ( codeword == "CONNECTFIRST" )
		{
			if ( chainid == 0 )
				throw CHAINS_ERROR ( "\"CHAIN\" expected before \"CONNECTFIRST\"." );

			CONNECT *connect=thischain->new_connectfirst();
			read_connect ( connect );
			read_bondgroup ( _CF_, "CF", connect );
		}
		else if ( codeword == "CHAINS" )
		{
			break;
		}
		else if ( codeword == "CLOSE" )
		{
			break;
		}
		else if ( codeword == "END_CROSS" )
		{
			break;
		}
		else if ( codeword == "GEOMCONSTRNR" )
		{
			break;
		}
		else if ( codeword == "PAUSE" )
		{
			// assert(thischain);
			// thischain->set_pause(true);
			pauseflag = true;
		}
		else
		{
			throw CHAINS_ERROR ( string ( "unknown or misplaced codeword \"" ) +codeword+"\"." );
		}
	}
	while ( sysu.nextfilerecord ( file ) == 0 );
	return first_chain;
}


/*!
    \fn CHAINS::chomp(string &str)
    \brief Removes carriage return symbol from the end of a string
 */
void CHAINS::chomp(string &_str)
{
	// workaround for bug in SYSUTILS
	string::size_type pos=_str.find ( "\n" );
	if ( pos != string::npos )
	_str.erase ( pos,_str.size()-pos );
	// workaround ends

}
