#include "bondinst.h"
#include <assert.h>
#include "mcgen.h"

BONDINST::BONDINST(BONDCLASS _class, const BONDDEF *_def, const ATOM *_atom1, const ATOM *_atom2, const ATOM *_atom3, const ATOM *_atom4, int _filerow)
{
bondclass=_class;
def=_def;
atom[0]=_atom1;
atom[1]=_atom2;
atom[2]=_atom3;
atom[3]=_atom4;
filerow=_filerow;
}

bool BONDINST::is_between_atoms(int _numatoms, ATOM *_atom[4])
{
bool straight=true;
bool reverse=true;
if (_numatoms != get_numatoms())
	return false;
for (int i=0; i<_numatoms; ++i)
{
	straight = straight and (atom[i] == _atom[i]);
	reverse = reverse and (atom[i] == _atom[_numatoms-i-1]);
}
return straight or reverse;
}

int BONDINST::calculate(ENERGY &_energy)
{
double energy_inc;
switch (bondclass)
{
case _BOND_:
	assert(atom[0] && atom[1]);
	energy_inc=def
		->bond_energy(ATOM::distance(atom[0],atom[1]));
	_energy.bonen+=energy_inc;
	if ((verbose_energy) and (abs(energy_inc) > BOND_ENERGY_LOG_THRESHOLD))
	{
		logfile.print(
				"Note: Bond (%s-%s) has positive energy %lf at field[%d].\n",
				atom[0]->get_label(),
				atom[1]->get_label(),
				energy_inc,
				filerow);
	}
	return 0;
case _ANGLE_:
        assert(atom[0] && atom[1] && atom[2]);
	energy_inc=def
		->angle_energy(ATOM::angle(atom[0],atom[1],atom[2]));
	_energy.angen+=energy_inc;
	if ((verbose_energy) && (abs(energy_inc)>ENERGY_LOG_THRESHOLD))
	{
		logfile.print(
		"Note: Angle (%s-%s-%s) has non-zero energy %lf at field[%d].\n",
		atom[0]->get_label(),
		atom[1]->get_label(),
		atom[2]->get_label(),
		energy_inc,
		filerow);
	}
	return 0;
case _DIHEDRAL_:
	assert(atom[0] && atom[1] && atom[2] && atom[3]);
	energy_inc=def->torsion_energy(
			ATOM::dihedral(atom[0],atom[1],atom[2],atom[3]));
	_energy.dihen+=energy_inc;
	{
		bool cou_needed=(def->get_scalar_cou() != 0);
		bool vdw_needed=(def->get_scalar_vdw() != 0);
		if (cou_needed or vdw_needed)
		{
			double distance=ATOM::distance(atom[0], atom[3]);
			if (cou_needed)
				_energy.couen+=def->get_scalar_cou() *
					ATOM::coulomb_energy
					(atom[0], atom[1], distance); 
			if (vdw_needed)
				_energy.vdwen+=def->get_scalar_vdw() *
					ATOM::vdw_energy
					(atom[0], atom[1], distance);
		}
		if ((verbose_energy) && (abs(energy_inc)>ENERGY_LOG_THRESHOLD))
		{
			logfile.print(
					"Note: Dihedral (%s-%s-%s-%s) has non-zero energy %lf at field[%d].\n",
					atom[0]->get_label(),
					atom[1]->get_label(),
					atom[2]->get_label(),
					atom[3]->get_label(),
					energy_inc,
					filerow);
		}
	}
	return 0;
default:
	assert(false);
}
return 0;
}

const ATOM *BONDINST::latest_atom() const
{
const ATOM *latest_a=atom[0];
int latest_n=latest_a->get_atoms_placement_order_id();
if (atom[1]->get_atoms_placement_order_id() > latest_n)
{
	latest_a=atom[1];
	latest_n=latest_a->get_atoms_placement_order_id();
}
if ((bondclass == _ANGLE_) or (bondclass == _DIHEDRAL_))
{
	if (atom[2]->get_atoms_placement_order_id() > latest_n)
	{
		latest_a=atom[2];
		latest_n=latest_a->get_atoms_placement_order_id();
	}
}
if (bondclass == _DIHEDRAL_)
{
	if (atom[3]->get_atoms_placement_order_id() > latest_n)
	{
		latest_a=atom[3];
		latest_n=latest_a->get_atoms_placement_order_id();
	}
}
return latest_a;
}

int BONDINST::get_filerow() const
{
return filerow;
}

bool BONDINST::is_similar(const BONDINST &_other, int _delta)
{
int numatoms=((int)bondclass)+1;

for (int i=0; i<numatoms; ++i)
{
	if (_other.atom[i]->get_oid() - atom[i]->get_oid() != _delta)
		return false;
}
return true;
}

int BONDINST::push_to_outlist(list<BOND_OUT> &_outlist,
		int _delta, int _repeat) const
{
BOND_OUT o;
o.bonddef=def;
//o.type=def->get_id()-1;
int numatoms=get_numatoms();
for (int i=0; i<numatoms; ++i)
{
	const ATOM *a=atom[i];
	assert(a);
	assert(a->get_has_coords());
	o.site[i]=a->get_oid();
}
_outlist.push_back(o);
return 1;
}

int BONDINST::get_numatoms() const
{
if (bondclass == _EXCLUDE_)
	return 2;
if (bondclass == _BOND_)
	return 2;
if (bondclass == _ANGLE_)
	return 3;
if (bondclass == _DIHEDRAL_)
	return 4;
assert(false);
}

BOND_OUT BONDINST::to_bond_out()
{
BOND_OUT o;
o.bonddef=def;
//o.type=def->get_id()-1;
o.numatoms=get_numatoms();
for (int i=0; i<o.numatoms; ++i)
{
	o.site[i]=atom[i]->get_oid();
}
o.constlen=0;
o.repeat=1;
o.spacing=0;
return o;
}

const BONDDEF *BONDINST::get_def() const
{
return def;
}


/*!
    \fn BONDINST::add_to_connection_lists()
    \brief Creates connections between every atom pairs in this bond.
 */
int BONDINST::add_to_connection_lists()
{
	int numatoms=get_numatoms();
	for ( int i=0; i<numatoms; ++i )
		for ( int j=0; j<numatoms; ++j )
		{
			ATOM *a = (ATOM*) atom[i];
			ATOM *b = (ATOM*) atom[j];
			a->save_connection ( b );
		}
	return 0;
}
