#include "bonddef.h"
#include "mcgen.h"

//========== constructor BONDDEF ==========

BONDDEF::BONDDEF()
{
id=0;
atom[0]=0;
atom[1]=0;
atom[2]=0;
atom[3]=0;
potkey=ZERO;
forcenr=0;
force=0;
filerow=0;
scalar_cou=0;
scalar_vdw=0;
excludes_longrange=true;
}

//========== destructor ~BONDDEF ==========

BONDDEF::~BONDDEF()
{
delete [] force;
}


//========== method set_forcenr ==========

int BONDDEF::set_forcenr(int _forcenr)
{
SYSUTILS tmp;
//#ifdef __DEBUG__
//logfile.print("set_forcenr(): forcenr=%d, _forcenr=%d\n",forcenr,_forcenr);
//#endif
assert(forcenr==0);
forcenr=_forcenr;

force=new double[forcenr];
tmp.allocate_error(force,"BONDDEF","FORCE array",_SYSUTILS_MEMERR);
return 0;
}


//========== method set_force ==========

int BONDDEF::set_force(int _nr, double _force)
{
assert(_nr>=1);
assert(_nr<=forcenr);

force[_nr-1]=_force;
return 0;
}

//========== method set_force ==========

int BONDDEF::set_force(int _nr, double _force, bool _deg2rad)
{
assert(_nr>=1);
assert(_nr<=forcenr);

if (_deg2rad)
	force[_nr-1]=MATHFUNC::deg2rad(_force);
else
	force[_nr-1]=_force;
return 0;
}


//========== method get_force ==========

double BONDDEF::get_force(int _nr) const
{
assert(_nr>=1);
if (_nr>forcenr)
	return 0;
return force[_nr-1];
}

//========== method set_atom ==========

int BONDDEF::set_atom(int _nr, const string &_id)
{
assert(_nr>=1);
assert(_nr<=4);
atom[_nr-1]=defs->get_atomarg(_id);
return 0;
}

//========== method set_potkey ==========

int BONDDEF::set_potkey(char *_potkey)
{
if (_potkey[0] == '-')
	excludes_longrange=false;

if (strcmp(_potkey,"zero")==0)
	potkey=ZERO;
else if (strcmp(_potkey,"harm")==0)
	potkey=HARM;
else if (strcmp(_potkey,"-hrm")==0)
	potkey=HARM;
else if (strcmp(_potkey,"cos")==0)
	potkey=COS;
else if (strcmp(_potkey,"sylv")==0)
	potkey=SYLV;
else if (strcmp(_potkey,"andi")==0)
	potkey=ANDI;
else if (strcmp(_potkey,"anpa")==0)
	potkey=ANPA;
else if (strcmp(_potkey,"cos5")==0)
	potkey=COS5;
else if (strcmp(_potkey,"cosj")==0)
	potkey=COSJ;
else if (strcmp(_potkey,"fmct")==0)
	potkey=FMCT;
else if (strcmp(_potkey,"ljpo")==0)
	potkey=LJPO;
else if (strcmp(_potkey,"-ljp")==0)
	potkey=LJPO;
else if (strcmp(_potkey,"12-6")==0)
	potkey=_12_6;
else if (strcmp(_potkey,"-126")==0)
	potkey=_12_6;
else if (strcmp(_potkey,"cosg")==0)
	potkey=COSG;
else if (strcmp(_potkey,"opa2")==0)
	potkey=OPA2;
else if (strcmp(_potkey,"fix")==0)
	potkey=FIX;
else if (strcmp(_potkey,"-fix")==0)
	potkey=FIX;
else throw(string("Unknow potkey name '")+_potkey+"'\n");
return 0;
}

//========== method get_potkey_str ==========

const char *BONDDEF::get_potkey_str() const
{
if (excludes_longrange)
	switch (potkey)
	{
		case ZERO: return "zero";
		case HARM: return "harm";
		case COS:  return "cos";
		case SYLV: return "sylv";
		case ANDI: return "andi";
		case ANPA: return "anpa";
		case COS5: return "cos5";
		case COSJ: return "cosj";
		case FMCT: return "fmct";
		case LJPO: return "ljpo";
		case COSG: return "cosg";
		case OPA2: return "opa2";
		case _12_6:return "12-6";
		case FIX:  return "fix";
		case COSB:  return "cosb";
		default:   return "????";
	}
else
	switch (potkey)
	{
		case HARM: return "-hrm";
		case LJPO: return "-ljp";
		case _12_6:return "-126";
		case FIX:  return "-fix";
		default:   return "????";
	}
}

//========== method get_atom ==========

const ATOMARG *BONDDEF::get_atom(int _nr) const
{
assert(_nr>=1);
assert(_nr<=4);
return atom[_nr-1];
}


//========== method set_id ==========

int BONDDEF::set_id(int _id) 
{
id=_id;
return 0;
}

//========== method get_id ==========

int BONDDEF::get_id() const
{
return id;
}

//========== method set_label ==========

int BONDDEF::set_label(const string &_label) 
{
label=_label;
return 0;
}

//========== method get_label ==========

string BONDDEF::get_label() const
{
return label;
}

//========== method get_excludes_longrange ==========

bool BONDDEF::get_excludes_longrange() const
{
return excludes_longrange;
}

//========== method set_filerow ==========

int BONDDEF::set_filerow(int _filerow) 
{
filerow=_filerow;
return 0;
}

//========== method get_filerow ==========

int BONDDEF::get_filerow() const
{
return filerow;
}

//========== method get_potkey ==========

Potkey BONDDEF::get_potkey() const
{
return potkey;
}

//========== method get_forcenr ==========

int BONDDEF::get_forcenr() const
{
return forcenr;
}

//========== method bond_energy ==========

double BONDDEF::bond_energy(const double _ell) const
{
//DEBUG
assert(this!=0);
if (forcenr==0)
{
	return 0;
}
if (forcenr!=2)
{
	ostringstream msg;
	msg<<"forcenr="<<forcenr<<", must be 2.";
	throw msg.str();
}
// END OF DEBUG
assert(forcenr==2);

//----BOND ENERGY CALCULATION FUNCTIONS ----
switch(potkey)
{
	case HARM:
		return 0.5*force[0]*pow(_ell-force[1],2);
	//--LENNARD JONES POTENTIAL
	//--2 PARAMETERS [EPSILON, SIGMA] (correspond to strength and radius)
	case LJPO:
		return 4*force[0]*(pow(force[1]/_ell,12)-pow(force[1]/_ell,6));
	//--LENNARD JONES POTENTIAL
	//--2 PARAMETERS: V_12, V_6
	case _12_6:
		return force[0]/pow(_ell,12)-force[1]/pow(_ell,6);
	// Fixed potential to offset energy values
	case FIX:
		return force[0];
	default:
		return 0;
}
}

//========== method bond_energy ==========
/*
double BONDDEF::bond_energy(double kei,double ell_0,double ell)
{
double tbonden;
tbonden=0.5*kei*pow(ell-ell_0,2);
//if (tbonden>0.0001)
//	logfile.print("ATOM bond_energy %lf %lf\n",ell,tbonden);
return tbonden;
};*/

//========== method angle_energy ==========

double BONDDEF::angle_energy(const double _theta) const
{
//logfile.print("angle_energy(%lf)\n",_theta);
assert(forcenr==2);
return angle_energy(force[1],_theta,force[0]);
}

//========== method angle_energy ==========

double BONDDEF::angle_energy(double theta_0,double theta,double kei)
{
// previous definition: tangen=0.5*kei*pow(cos(theta)-cos(theta_0),2);
double tangen;

tangen=0.5*kei*pow(theta-theta_0,2);
//if (tangen>0.001)
//	logfile.print("ATOM angle_energy %10lf-%10lf => %10lf\n",
//		theta*180/M_PI,theta_0*180/M_PI,tangen);
return tangen;
};

//========== method torsion_energy ==========

double BONDDEF::torsion_energy(const double _tau) const
{

// Heiki: removed
//if (forcenr!=DIHFORCENR)
//{
//	ostringstream message;
//	message << "Torsion energy must have " << DIHFORCENR <<" parameters. " << forcenr
//		<<" was given instead.";
//	throw message.str();
//}

//return torsion_energy(potkey,_tau,
//	force[0],force[1],force[2],force[3],force[4],force[5],force[6]);


double ttoren=0.0;

switch(potkey)
{
	case COS:
	{
		ttoren=force[0]*(1.0+cos(force[2]*(_tau)-force[1]));
		break;
	}
	case SYLV:
	{
		for(int ti=0;ti<7;ti++)
			ttoren += force[ti+2]*pow(-1.0,ti)*pow(cos(_tau),ti);
		break;
	}
	case COS5:
	{
		for(int ti=0;ti<4;ti++)
			ttoren += 0.5*force[ti+2]*(1.0-cos((ti+1)*_tau));
		break;
	}
	case COSJ:
	{
		for(int ti=0;ti<3;ti++)
			ttoren += 0.5*force[ti+2]*(1.0-force[ti+5]*cos((ti+1)*_tau));
//		for(int ti=0;ti<forcenr-3;ti=ti+4)
//			ttoren += 0.5*force[ti+2]*(1.0-force[ti+5]*cos((ti+1)*_tau));

		break;
	}
	//FMCT Fourier MultiComponent Torsion (dynamic size) -- Added by Alar Ainla
	case FMCT:
	{
		//Parameters are given V1, d1, V2, d2, etc.
		for(int ti=0;ti<forcenr-3;ti+=2)
			ttoren += 0.5*force[ti+2]*(1.0-force[ti+3]*cos((ti/2+1)*_tau));
		//	ttoren += 0.5*force[ti+0]*(1.0-force[ti+1]*cos((ti/2+1)*_tau));
		break;
	}
	
	case ANDI:
	{
		for(int ti=0;ti<7;ti++)
			ttoren += force[ti+2]*(cos((ti+1)*_tau)+1.0);
		break;
	}
	case ANPA:
	{
		ttoren = force[0];
		for(int ti=1;ti<=3;ti++)
			ttoren += force[2*ti+1]*cos(ti*_tau) + 
				force[2*ti+2]*sin(ti*_tau);
		break;
	}
	case COSG:
	{
		for(int ti=0;ti<forcenr-1;ti+=2)
			ttoren=ttoren+force[ti]*cos(force[ti+1]*_tau);
		break;
	}
	case OPA2:
	{
		ttoren=ttoren+force[2]+
			force[3]*(1+cos(_tau))+
			force[4]*(1-cos(2*_tau))+
			force[5]*(1+cos(3*_tau))+
			force[6]*(1-cos(4*_tau));
		break;
	}
	default:	ttoren=0.0;
}

/* Heiki: replaced by switch()
double ttoren=0.0;

if(potkey==SYLV)
{
	for(int ti=0;ti<7;ti++)
		ttoren += force[ti]*pow(-1.0,ti)*pow(cos(_tau),ti);
}
else if(potkey==ANDI)
{
	for(int ti=0;ti<7;ti++)
		ttoren += force[ti]*(cos((ti+1)*_tau)+1);
}
else if(potkey==ANPA)
{
	ttoren = force[0];
	for(int ti=1;ti<=3;ti++)
		ttoren += force[2*ti-1]*cos(ti*_tau) + force[2*ti]*sin(ti*_tau);
}
*/

//logfile.print("torsion_energy %lf %lf\n",tau,ttoren);
return ttoren;
}

/*
//========== method torsion_energy ==========

double BONDDEF::torsion_energy(Potkey _potkey,double tau,double a0,double a1,double a2,
                             double a3,double a4,double a5,double a6)
{

//potkey: 1 - Sylvie
//        4 - Andi

int ti;
double ttoren=0.0,ta[7];
ta[0]=a0;
ta[1]=a1;
ta[2]=a2;
ta[3]=a3;
ta[4]=a4;
ta[5]=a5;
ta[6]=a6;
if(_potkey==SYLV)
{
	for(ti=0;ti<7;ti++)
	{
		ttoren=ttoren+ta[ti]*pow(-1.0,ti)*pow(cos(tau),ti);
	}
}

if(_potkey==ANDI)
{
	for(ti=0;ti<7;ti++)
	{
		ttoren=ttoren+ta[ti]*(cos((ti+1)*tau)+1);
	}
}
//logfile.print("torsion_energy %lf %lf\n",tau,ttoren);
return ttoren;
};
*/

//========== method get_scalar_cou ==========

double BONDDEF::get_scalar_cou() const
{
return scalar_cou;
}

//========== method get_scalar_vdw ==========

double BONDDEF::get_scalar_vdw() const
{
return scalar_vdw;
}

//========== method set_scalar_cou ==========

int BONDDEF::set_scalar_cou(double _scalar_cou)
{
//printf("scalar_cou=%lf\n",_scalar_cou);
scalar_cou=_scalar_cou;
return 0;
}

//========== method set_scalar_vdw ==========

int BONDDEF::set_scalar_vdw(double _scalar_vdw)
{
//printf("scalar_vdw=%lf\n",_scalar_vdw);
scalar_vdw=_scalar_vdw;
return 0;
}

//========== method dlpoly_potparamnr ==========

int BONDDEF::get_dlpoly_paramnr() const
{
if (potkey == SYLV) return 7;
//if (potkey == COSB) return 11;
//if (potkey == TORV) return 7;
if (potkey == COS5) return 5;
if (potkey == ZERO) return 0;
if (potkey == ANDI) return 7;
//if (potkey == OPA2) return 7;
return 5;
}

//========== method dlpoly_potparamnr ==========

int BONDDEF::get_if_dlpoly_needs_scalefactors() const
{
if (potkey == SYLV) return 1;
//if (potkey == COSB) return 1;
//if (potkey == TORV) return 1;
if (potkey == COS5) return 1;
if (potkey == ZERO) return 1;
if (potkey == ANDI) return 1;
if (potkey == OPA2) return 1;
return 0;
}

BONDDEF BONDDEF::make_dlpoly_compatible(BONDCLASS _bondclass) const
{
BONDDEF o;
o.atom[0]=atom[1];
o.atom[1]=atom[2];
o.atom[2]=atom[2];
o.atom[3]=atom[3];
o.excludes_longrange=excludes_longrange;
o.set_label(label);
if (_bondclass == _ANGLE_)
{
	o.potkey=potkey;
	o.set_forcenr(forcenr);
	for (int i=0; i<forcenr; ++i)
		o.force[i]=force[i];
	assert(forcenr >= 2);
	o.force[1]=MATHFUNC::rad2deg(force[1]);
}
else if ((_bondclass == _DIHEDRAL_) && (potkey==FMCT))
{
	o.potkey=COSB;
	o.set_forcenr(13);
	for (int i=0; i<o.forcenr; ++i)
		o.force[i]=0;
	o.force[0]=scalar_cou;
	o.force[1]=scalar_vdw;
	for(int ti=0;ti<forcenr-3;ti+=2)
	{
		o.force[2] += force[ti+2];
		o.force[ti/2+3] = -0.5*force[ti+2]*force[ti+3];
	}
}
else if ((_bondclass == _DIHEDRAL_) && (potkey==COSJ))
{
	o.potkey=COSB;
	o.set_forcenr(13);
	for (int i=0; i<o.forcenr; ++i)
		o.force[i]=0;
	o.force[0]=scalar_cou;
	o.force[1]=scalar_vdw;

	for(int ti=0;ti<3;ti++)
	{
		o.force[2] += force[ti+2];
		o.force[3+ti] = -0.5*force[ti+2]*force[ti+5];
	}
}
else
{
	o.potkey=potkey;
	o.set_forcenr(forcenr);
	for (int i=0; i<forcenr; ++i)
		o.force[i]=force[i];
}
return o;
}

