#include "atom.h"
#include "unit.h"
#include "mcgen.h"

list<ATOM*> ATOM::atoms_placement_order;
list<ATOM*>::iterator ATOM::next_placement=atoms_placement_order.begin();
int ATOM::placed_count=0;

//========== constructor ATOM ==========

ATOM::ATOM()
{
set_has_coords(false);
atomdef=0;
atomarg=0;
con_meth=NOPLACE;
con_atom[0]=0;
con_atom[1]=0;
con_atom[2]=0;
con_const=0;
con_cang=0;
con_cdih=0;
atoms_placement_order_id=0;
}

//========== method add_to_atoms_placement_order ==========

int ATOM::add_to_atoms_placement_order()
{
//logfile.print("DEBUG: ATOM %p->add_to_atoms_placement_order()\n",this);
if (atoms_placement_order_id != 0)
		return atoms_placement_order_id;
if (!atomdef)
{
	con_meth=RAND;
			assert(unit->get_unitclass() & (UC_SEED|UC_TAIL));
}
else if (atomdef->get_has_coords())
{
	con_meth=FIXED;
}
else
{
		for (int i=1;i<=3;i++)
		{
			int A=atomdef->get_atomXid(i);
			int U=atomdef->get_unitXid(i);
			ATOM *p_atom=unit->relative_unit(U)->get_atom(A);
			con_atom[i-1]=p_atom;
			if (p_atom->atoms_placement_order_id==0)
			{
				p_atom->add_to_atoms_placement_order();
			}
		}

		con_const=atomdef->get_const();
		con_cang=atomdef->get_cang();
		con_cdih=atomdef->get_cdih();
		con_meth=atomdef->get_method();

		switch (con_meth)
		{
	case HYDRO:
			con_atom[1]->immediately_connected_atoms.insert(this);
			immediately_connected_atoms.insert(con_atom[1]);
			break;
	case BACK:
			con_atom[2]->immediately_connected_atoms.insert(this);
			immediately_connected_atoms.insert(con_atom[2]);
			break;
		default:
			;
		// default: independent atom, not connected to anything
		}
}
bool islast=(next_placement==atoms_placement_order.end());
atoms_placement_order.push_back(this);
atoms_placement_order_id=atoms_placement_order.size();
if (islast)
{
		next_placement=atoms_placement_order.end();
		--next_placement;
}

if ((atomarg) && (atomdef) && (not atomdef->get_has_coords()))
{
		save_connections(con_atom);
}
else
{
		save_connection ( this );
}

return atoms_placement_order_id;
}

//========== static method place_next ==========

int ATOM::place_next(ENERGY &_energy)
{
assert(next_placement != atoms_placement_order.end());
ATOM *t=*next_placement;
++next_placement;
int status=t->place();
if (not t->satisfies_geomconstraints())
		return _ATOM_place_GEOMFAIL_;
if (t->has_overlaps())
		return _ATOM_place_GEOMFAIL_;
t->add_energy(_energy);
return status;
}

//========== method place ==========

int ATOM::place()
{
//logfile.print("DEBUG: ATOM %p->place()\n",this);
switch (con_meth)
{
case BACK:
case HYDRO:
		calcatom(con_atom, con_const, con_cang, con_cdih, con_meth);
		break;
case RAND:
		return place_randomly();
case FIXED:
	set_coords(atomdef);
		break;
default:
		assert(false);
}
assert(get_has_coords());
//logfile.print("DEBUG: Placed atom at (%lf, %lf, %lf)\n",get_xcoord(),get_ycoord(),get_zcoord());
history.push(this); // save ATOM history record
return _ATOM_place_OK_;
}

//========== method place_randomly ==========

int ATOM::place_randomly()
{
double a[3],x[3];
int i,j;

assert(!atomdef);
assert(!get_has_coords());

for (i=0;i<3;i++)
		a[i]=randome->RANDOME_output_random_number();

for (i=0;i<3;i++)
{
		x[i]=0;
		for (j=0;j<3;j++)
			x[i]+=a[j]*params->get_cell(j+1,i+1);
}

set_coords(x[0],x[1],x[2]);

history.push(this); // save ATOM history record
return _ATOM_place_OK_;
}

//========== method calcatom ==========

int ATOM::calcatom(ATOM *_dep_atoms[3],
		CONSTDEF *_bon, CONSTDEF *_ang, CONSTDEF *_dih, AtomMethod _method)
{
double x[3], y[3], z[3];
for (int i=0;i<3;i++)
{
		assert(_dep_atoms[i]->get_has_coords());
		x[i]=_dep_atoms[i]->get_xcoord();
		y[i]=_dep_atoms[i]->get_ycoord();
		z[i]=_dep_atoms[i]->get_zcoord();
}
double bon=_bon->get_random_bond();
double ang=_ang->get_random_angle();
double dih=_dih->get_random_dihedral();
if (_method==HYDRO)
{
	calchydro(x[2],y[2],z[2], x[1],y[1],z[1], x[0],y[0],z[0],
		bon,MATHFUNC::rad2deg(dih),MATHFUNC::rad2deg(ang));
}
else
{
	calcback(x[2],y[2],z[2], x[1],y[1],z[1], x[0],y[0],z[0],
		bon,MATHFUNC::rad2deg(dih),MATHFUNC::rad2deg(ang));
}
assert (! connected_atoms.empty());
//if (abs(bon-distance(this,previousatom[0]))>0.000001)
//	logfile.print("Distance must be %lf, is %lf\n",bon,distance(this,previousatom[0]));
return 0;
}

//========== method calcback ==========

void ATOM::calcback(double x1,double y1,double z1,double x2,double y2,
                     double z2,double x3,double y3,double z3,double bond,
                     double dihan,double alpha)
{
//logfile.print("calcback: (%lf,%lf,%lf),(%lf,%lf,%lf),(%lf,%lf,%lf),%lf,%lf,%lf\n",
//	x1,y1,z1,x2,y2,z2,x3,y3,z3,bond,dihan,alpha);

double tx,ty,tz,ax,ay,az,cx,cy,cz,dx,dy,dz,ksii,gamma,beta;

tx=x2;
ty=y2;
tz=z2;

//========== Origin shift ==========
cx=x1-tx;
cy=y1-ty;
cz=z1-tz;
ax=x3-tx;
ay=y3-ty;
az=z3-tz;
//========== Rotate across z axis by ksii ==========
if(cx==0.0 && cy==0.0) {
	cx=cy=1.0E-30;
}
ksii=acos(cx/sqrt(pow(cx,2)+pow(cy,2)));
ksii=mat.rad2deg(ksii);
//logfile.print("%lf\n",ksii);
if(cy<0) {
	ksii=360.0-ksii;
}
mat.mathinpradxyz(mat.deg2rad(ksii),cx,cy,cz);
mat.rotatez();
cx=mat.mathoutx();
cy=mat.mathouty();
cz=mat.mathoutz();
mat.mathinpradxyz(mat.deg2rad(ksii),ax,ay,az);
mat.rotatez();
ax=mat.mathoutx();
ay=mat.mathouty();
az=mat.mathoutz();
//========== Rotate across new y axis by gamma ==========
if(cx==0.0 && cz==0.0) {
	cx=cz=1.0E-30;
}
gamma=acos(cx/sqrt(pow(cx,2)+pow(cz,2)));
gamma=mat.rad2deg(gamma);
if(cz<0) {	
	gamma=360.0-gamma;
}
mat.mathinprad(mat.deg2rad(gamma));
mat.rotatey();
ax=mat.mathoutx();
ay=mat.mathouty();
az=mat.mathoutz();
mat.mathinpradxyz(mat.deg2rad(gamma),cx,cy,cz);
mat.rotatey();
cx=mat.mathoutx();
cy=mat.mathouty();
cz=mat.mathoutz();
//========== Rotate across new-new x axis by beta ==========
if(ay==0.0 && az==0.0) {
	ay=az=1.0E-30;
}
beta=acos(ay/sqrt(pow(ay,2)+pow(az,2)));
beta=mat.rad2deg(beta);
if(az<0) {
	beta=360.0-beta;
}
beta=beta+dihan;
mat.mathinprad(mat.deg2rad(beta));
mat.rotatex();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
alpha=180.0-alpha;
dx=dx+bond*cos(mat.deg2rad(alpha));
dy=bond*sin(mat.deg2rad(alpha));
mat.mathinpradxyz(mat.deg2rad(beta),dx,dy,dz);
mat.rotatexinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
mat.mathinprad(mat.deg2rad(gamma));
mat.rotateyinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
mat.mathinprad(mat.deg2rad(ksii));
mat.rotatezinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
set_coords(dx+tx, dy+ty, dz+tz);
//logfile.print("otse %lf %lf %lf\n",coord[0],coord[1],coord[2]);

};

//========== method calchydro==========

void ATOM::calchydro(double x1,double y1,double z1,double x2,double y2,
                      double z2,double x3,double y3,double z3,double bond,
                      double dihan,double alpha)
{

double tx,ty,tz,ax,ay,az,cx,cy,cz,dx,dy,dz,ksii,gamma,beta;

tx=x2;
ty=y2;
tz=z2;
//========== Origin shift ==========
cx=x1-tx;
cy=y1-ty;
cz=z1-tz;
ax=x3-tx;
ay=y3-ty;
az=z3-tz;
//logfile.print("0a: %lf %lf %lf\n",ax,ay,az);
//logfile.print("0c: %lf %lf %lf\n",cx,cy,cz);
//=========== Rotate across z axis by ksii ==========
if(cx==0.0 && cy==0.0) {
	cx=cy=1.0E-30;
}
ksii=acos(cx/sqrt(pow(cx,2)+pow(cy,2)));
ksii=mat.rad2deg(ksii);
//logfile.print("%lf\n",ksii);
if(cy<0) {
	ksii=360.0-ksii;
}
mat.mathinpradxyz(mat.deg2rad(ksii),cx,cy,cz);
mat.rotatez();
cx=mat.mathoutx();
cy=mat.mathouty();
cz=mat.mathoutz();
mat.mathinpradxyz(mat.deg2rad(ksii),ax,ay,az);
mat.rotatez();
ax=mat.mathoutx();
ay=mat.mathouty();
az=mat.mathoutz();
//logfile.print("1a: %0.10lf %0.10lf %0.10lf ksii %lf\n",ax,ay,az,ksii);
//logfile.print("1c: %0.10lf %0.10lf %0.10lf ksii %lf\n",cx,cy,cz,ksii);
//========== Rotate across new y axis by gamma ==========
if(cx==0.0 && cz==0.0) {
	cx=cz=1.0E-30;
}
gamma=acos(cx/sqrt(pow(cx,2)+pow(cz,2)));
gamma=mat.rad2deg(gamma);
//logfile.print("%lf\n",gamma);
if(cz<0) {
	gamma=360.0-gamma;
}
mat.mathinprad(mat.deg2rad(gamma));
mat.rotatey();
ax=mat.mathoutx();
ay=mat.mathouty();
az=mat.mathoutz();
mat.mathinpradxyz(mat.deg2rad(gamma),cx,cy,cz);
mat.rotatey();
cx=mat.mathoutx();
cy=mat.mathouty();
cz=mat.mathoutz();
//logfile.print("2a: %0.10lf %0.10lf %0.10lf gamma %lf\n",ax,ay,az,gamma);
//logfile.print("2c: %0.10lf %0.10lf %0.10lf gamma %lf\n",cx,cy,cz,gamma);
//========== Rotate across new-new x axis by beta ==========
if(ay==0.0 && az==0.0) {
	ay=az=1.0E-30;
}
beta=acos(ay/sqrt(pow(ay,2)+pow(az,2)));
beta=mat.rad2deg(beta);
//logfile.print("%lf\n",beta);
if(az<0) {
	beta=360.0-beta;
}
//dihan=180-dihan;
beta=beta+180.0;//dihan;
mat.mathinprad(mat.deg2rad(beta));
mat.rotatex();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
//logfile.print("3a: %0.10lf %0.10lf %0.10lf beta %lf\n",ax,ay,az,beta);
//logfile.print("3c: %0.10lf %0.10lf %0.10lf beta %lf\n",dx,dy,dz,beta);
//dihan=180.0-dihan;
//alpha=180.0-alpha;
dz=bond*sin(mat.deg2rad(dihan));
//logfile.print("dihan %lf\n",dihan);
ax=bond*cos(mat.deg2rad(dihan));
//logfile.print("ax %lf\n",ax);
dx=ax*cos(mat.deg2rad(alpha));
dy=ax*sin(mat.deg2rad(alpha));
mat.mathinpradxyz(mat.deg2rad(beta),dx,dy,dz);
//logfile.print("c4: %0.10lf %0.10lf %0.10lf alpha %lf\n",dx,dy,dz,alpha);
mat.rotatexinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
//logfile.print("c5: %0.10lf %0.10lf %0.10lf\n",dx,dy,dz);
mat.mathinprad(mat.deg2rad(gamma));
mat.rotateyinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
//logfile.print("c6: %0.10lf %0.10lf %0.10lf\n",dx,dy,dz);
mat.mathinprad(mat.deg2rad(ksii));
mat.rotatezinv();
dx=mat.mathoutx();
dy=mat.mathouty();
dz=mat.mathoutz();
//logfile.print("c7: %0.10lf %0.10lf %0.10lf\n",dx,dy,dz);
set_coords(dx+tx, dy+ty, dz+tz);
//logfile.print("c8: %0.10lf %0.10lf %0.10lf\n",coord[0],coord[1],coord[2]);
ax=sqrt(pow(get_xcoord()-x2,2)+pow(get_ycoord()-y2,2)+pow(get_zcoord()-z2,2));
//logfile.print("%lf\n",ax);
};
/**/
//========== ==========

double ATOM::vdw_energy(const ATOM *_atom1, const ATOM *_atom2, double _distance)
{
double result=vdw->find(_atom1->get_atomarg()->get_id(),
	_atom2->get_atomarg()->get_id())->
	vdw_energy(_distance);
if (verbose_energy)
{
if (abs(result)>VDW_ENERGY_LOG_THRESHOLD)
	logfile.print("<VDW %s-%s %s-%s:%0.3lf=%0.3lf>\n",
		_atom1->get_location().c_str(),_atom2->get_location().c_str(),
		_atom1->get_atomarg()->get_label(),
		_atom2->get_atomarg()->get_label(),
		distance(_atom1,_atom2),result);
}
return result;
}

//========== ==========

double ATOM::coulomb_energy(double qi,double qj,double ln1,double cconst)
{
double tcoulen;

//Heiki 010205
if(ln1==0.0)
	tcoulen=0.0;
else
	tcoulen=(qi*qj*cconst)/ln1;
//endHeiki
//logfile.print("DEBUG ATOM_COULOMB %12.3lf %12.3lf\n",ln1,tcoulen);
return tcoulen;
};

double ATOM::coulomb_energy(const ATOM *_atom1, const ATOM *_atom2, double _distance)
{
double result=coulomb_energy(_atom1->get_charge(),_atom2->get_charge(),
	_distance,COULOMBCONST);
//if ((_atom1->get_atomarg()->get_id()==11) and (_atom2->get_atomarg()->get_id()==15)) {
//	logfile.print("HEY!\n");
//}
if (verbose_energy)
{
if (abs(result)>COULOMB_ENERGY_LOG_THRESHOLD)
	logfile.print("<COU %s-%s %s-%s:%0.3lf=%0.3lf>\n",
		_atom1->get_location().c_str(),_atom2->get_location().c_str(),
		_atom1->get_atomarg()->get_label(),
		_atom2->get_atomarg()->get_label(),
		distance(_atom1,_atom2),result);
}
return result;
}

//========== ==========

double ATOM::energy_exp(double energy,double temperature)
{
double tenexp;
tenexp=exp(-energy*KCALMOL2JOULE/(BC*temperature));
//logfile.print("Energy_exp %lf %le\n",energy,tenexp);
return tenexp;
};

double ATOM::energy_exp(double energy)
{
return energy_exp(energy,params->get_temperature());
};

//========== ===========

//========== method set_atomarg ==========

int ATOM::set_atomarg(string _atomname)
{
atomarg=defs->get_atomarg(_atomname);
if (atomarg)
	return 0;
else
{
	ostringstream msg;
	msg<<"Error: unknown atom "<<_atomname;
	string exc=msg.str();
	throw exc;
}
}

//========== method set_atomdef ==========

int ATOM::set_atomdef(ATOMDEF *_atomdef)
{
assert(_atomdef != 0);
assert(_atomdef->get_atomarg() != 0);
atomdef=_atomdef;
atomarg=atomdef->get_atomarg();
return 0;
}

//========== method get_atomdef ==========

ATOMDEF *ATOM::get_atomdef() const
{
return atomdef;
}

//========== method get_atomarg ==========

ATOMARG *ATOM::get_atomarg() const
{
return atomarg;
}

//========== method set_unit ==========

int ATOM::set_unit(UNIT *_unit)
{
unit=_unit;
return 0;
}

//========== method get_unit ==========

UNIT *ATOM::get_unit() const
{
return unit;
}

//========== method set_index ==========

int ATOM::set_index(int _index)
{
index=_index;
return 0;
}

//========== method get_index ==========

int ATOM::get_index() const
{
return index;
}

//========== method distance_from_atom ==========

double ATOM::distance_from_atom(const ATOM &_atom2) const
{
return distance(this,&_atom2);
}

//========== method distance ==========

double ATOM::distance(const ATOM *_atom1, const ATOM *_atom2)
{
double dx=_atom1->get_xcoord()-_atom2->get_xcoord();
double dy=_atom1->get_ycoord()-_atom2->get_ycoord();
double dz=_atom1->get_zcoord()-_atom2->get_zcoord();
image(params->get_imcon(),0,1,1,params->get_cell(),&dx,&dy,&dz);
	double result=sqrt(dx*dx+dy*dy+dz*dz);
//logfile.print("Distance is %lf\n",result);
return result;
}

//========== method angle ==========

double ATOM::angle(const ATOM *_atom1, const ATOM *_atom2, const ATOM *_atom3)
{
double dx1=_atom1->get_xcoord()-_atom2->get_xcoord();
double dy1=_atom1->get_ycoord()-_atom2->get_ycoord();
double dz1=_atom1->get_zcoord()-_atom2->get_zcoord();
double dx3=_atom3->get_xcoord()-_atom2->get_xcoord();
double dy3=_atom3->get_ycoord()-_atom2->get_ycoord();
double dz3=_atom3->get_zcoord()-_atom2->get_zcoord();
image(params->get_imcon(),0,1,1,params->get_cell(),&dx1,&dy1,&dz1);
image(params->get_imcon(),0,1,1,params->get_cell(),&dx3,&dy3,&dz3);
double result=MATHFUNC::dotprodang(dx1,dy1,dz1,dx3,dy3,dz3);
//logfile.print("Angle is %lf\n",result);
return result;
}

//========== method dihedral ==========

double ATOM::dihedral(const ATOM *_atom1, const ATOM *_atom2, const ATOM *_atom3, const ATOM *_atom4)
{
double dx1=_atom1->get_xcoord()-_atom2->get_xcoord();
double dy1=_atom1->get_ycoord()-_atom2->get_ycoord();
double dz1=_atom1->get_zcoord()-_atom2->get_zcoord();
double dx2=_atom3->get_xcoord()-_atom2->get_xcoord();
double dy2=_atom3->get_ycoord()-_atom2->get_ycoord();
double dz2=_atom3->get_zcoord()-_atom2->get_zcoord();
double dx3=-dx2;
double dy3=-dy2;
double dz3=-dz2;
double dx4=_atom4->get_xcoord()-_atom3->get_xcoord();
double dy4=_atom4->get_ycoord()-_atom3->get_ycoord();
double dz4=_atom4->get_zcoord()-_atom3->get_zcoord();
image(params->get_imcon(),0,1,1,params->get_cell(),&dx1,&dy1,&dz1);
image(params->get_imcon(),0,1,1,params->get_cell(),&dx2,&dy2,&dz2);
image(params->get_imcon(),0,1,1,params->get_cell(),&dx3,&dy3,&dz3);
image(params->get_imcon(),0,1,1,params->get_cell(),&dx4,&dy4,&dz4);
MATHFUNC m;
double result=m.vectprodvectproddotprodang(dx1,dy1,dz1,dx2,dy2,dz2,
	dx3,dy3,dz3,dx4,dy4,dz4);
//logfile.print("Dihedral is %lf\n",result);
return result;
}


//========== method overlaps_with ==========

bool ATOM::overlaps_with(ATOM &_atom2) // Note: "_atom2" has to be an elder one
{				       // and "this" the newly placed one.
assert(this);
assert(&_atom2);
assert(&_atom2 != this);
if (not get_has_coords())
	return false;
if (not _atom2.get_has_coords())
	return false;
if (not (unit->get_unitclass() & (UC_NORMAL|UC_PRESET)))
	return false;
if (not (_atom2.unit->get_unitclass() & (UC_NORMAL|UC_PRESET)))
	return false;
if (_atom2.atomarg->is_invisible_for(atomarg))
	return false;
double dist=distance_from_atom(_atom2);
if (dist < get_atomarg()->get_radius()+
		_atom2.get_atomarg()->get_radius())
{
	if (verbose_geom)
	{
		logfile.print("Note: atoms %s - %s overlap: %lf < %lf + %lf\n",
				get_location().c_str(),
				_atom2.get_location().c_str(),
				dist,
				get_atomarg()->get_radius(),
				_atom2.get_atomarg()->get_radius());
		double dx=get_xcoord()-_atom2.get_xcoord();
		double dy=get_ycoord()-_atom2.get_ycoord();
		double dz=get_zcoord()-_atom2.get_zcoord();
		image(params->get_imcon(),0,1,1,params->get_cell(),&dx,&dy,&dz);
		logfile.print("Note: Delta vector is: (%lf, %lf, %lf)\n", dx, dy, dz);
		logfile.print("%s %lf %lf %lf\n", get_label(),
				get_xcoord(),get_ycoord(),get_zcoord());
		logfile.print("%s %lf %lf %lf\n", _atom2.get_label(),
				_atom2.get_xcoord(),_atom2.get_ycoord(),_atom2.get_zcoord());
		list<ATOM*>::iterator it=atoms_placement_order.begin();
		GENXYZ e;
		for (; it != next_placement; ++it)
			if ((*it)->get_atomarg())
				e.push_back(*it);
		e.write_xyz("geomerr.xyz");
	}
	return true;
}
return false;
}

//========== method get_label ==========

const char *ATOM::get_label() const
{
assert(atomarg != 0);
assert(get_atomarg()->get_label()!=0);
return get_atomarg()->get_label();
}

//========== method take_back_one_atom ==========
ATOM *ATOM::take_back_one_atom()
{
if (history.empty())
{
	logfile.print("History empty!\n");
	return 0;
}
ATOM *atom=history.top();
history.pop();
--next_placement;
assert(*next_placement == atom);
//logfile.print("History pop(%d)\n", (atom)?(1):(0) );
if (atom)
{
	assert(atom->get_has_coords());
	atom->set_has_coords(false);
	atom->unit->set_has_coords(false);
}
return atom;
}

//========== method has_previousatom ==========
//
//bool ATOM::has_previousatom(ATOM *_previousatom)
//{
//return (previousatom[0]==_previousatom)
//	|| (previousatom[1]==_previousatom)
//	|| (previousatom[2]==_previousatom);
//}

//========== method has_connection_with ==========

bool ATOM::has_connection_with(ATOM *_otheratom)
{
return connected_atoms.count(_otheratom);
}

/*//========== method get_connections ==========

string ATOM::get_connections()
{
ostringstream output;
set<ATOM*>::iterator i,end;
for (i=connected_atoms.begin(),end=connected_atoms.end();i<end;i++)
{
	output<<'['<<get_locaton();
	output<<'-'<<(i->get_locaton())<<"] ";
}
return output.str();
}*/

//========== method get_location ==========

string ATOM::get_location() const
{
if (atomdef)
{
	ostringstream output;
	output<<unit->get_position()<<get_label()<<" "<<get_index();
	return output.str();
}
else
{
	return "no_atomdef";
}
}

//========== method save_connection ==========

void ATOM::save_connection(ATOM *_atom2)
{
if (connected_atoms.count(_atom2) == 0)
{
	connected_atoms.insert(_atom2);
#ifdef DEBUG_CONNECTIONS
	logfile.print("Connection: %s-%s\n",
		get_location().c_str(),
		_atom2->get_location().c_str());
#endif
}
}

//========== method get_previousatom ==========
//
//ATOM *ATOM::get_previousatom()
//{
//return previousatom[0];
//}

//========== method get_charge =========

double ATOM::get_charge() const
{
return get_atomarg()->get_charge();
}

//========== method get_radius =========

double ATOM::get_radius() const
{
return get_atomarg()->get_radius();
}

//========== method get_atomid =========

int ATOM::get_atomid() const
{
return get_atomarg()->get_id();
}

//========== method satisfies_geomconstraints ==========

bool ATOM::satisfies_geomconstraints()
{
int i;
if (con_meth == FIXED)
	return true;
MOLECULE *mol=unit->get_molecule();
for (i=mol->get_geomconstrnr();i>=1;i--)
	if (mol->get_geomconstr(i)->satisfied(this) == false)
	{
		if (verbose_geom)
			logfile.print("Note: ATOM %s does not satisfy geometric constraints\n",
					get_location().c_str());
		return false;
	}
return true;
}

//========== GEOMETRY CHECKS ==========

bool ATOM::has_overlaps()
{
assert(get_has_coords());
list<ATOM*>::iterator begin, it;
begin=atoms_placement_order.begin();
it=next_placement;
if (it == begin)
	return 0;
if (unit->get_unitclass() & UC_PRESET)
	return 0;
do
{
	--it;
	ATOM *otheratom=*it;
	assert(otheratom->get_has_coords());
	UNIT *otherunit=otheratom->get_unit();
	
	if (otherunit == unit) // inside one unit
	{
		if (unit->get_unitclass() & UC_BEGIN)
		{
			if (unit->get_unitdef()->get_genmeth()==COORDS) // No geometry check inside units
				continue;               // that are given by exact coordinates.
		}
	}

	if (not (otherunit->get_unitclass() & UC_REAL))
		continue;

	if (this->has_connection_with(otheratom))
		continue;

	if (this->overlaps_with(*otheratom))
		return true; // Found overlap
}
while (it != begin);

return false;
}

//========== method restore_connected_atoms ==========

int ATOM::restore_connected_atoms()
{
ATOM *atom_p[3];

if (! atomdef)
	return 0;

//if (not get_has_coords())
//	return 0; // Place will set it fine anyway!

for (int i=1;i<=3;i++)
{
	int A=atomdef->get_atomXid(i);
	int U=atomdef->get_unitXid(i);
	atom_p[i-1]=unit->relative_unit(U)->get_atom(A);
}

return 0;
}

//========== method get_number_of_connected_atoms ==========

unsigned int ATOM::get_number_of_connected_atoms()
{
return connected_atoms.size();
}

//========== method set_oid ==========

int ATOM::set_oid(int _oid)
{
oid=_oid;
return 0;
}

//========== method get_oid ==========

int ATOM::get_oid() const
{
return oid;
}

//========== method get_placed_count ==========

int ATOM::get_placed_count()
{
list<ATOM*>::iterator it=next_placement;
if (it == atoms_placement_order.begin())
{
	return 0;
}
else
{
	--it;
	return (*it)->atoms_placement_order_id;
}
}

//========== method get_atoms_placement_order_id ==========

int ATOM::get_atoms_placement_order_id() const
{
return atoms_placement_order_id;
}

//========== method add_to_forcelist ==========

int ATOM::add_to_forcelist(const BONDINST &_force)
{
forcelist.push_back(_force);
if(verbose_energy)
logfile.print("Note: atom %s has force from field[%d]\n",get_location().c_str(),_force.get_filerow());
return 0;
}

//========== method add_energy ==========

int ATOM::add_energy(ENERGY &_energy)
{
if (not (unit->get_unitclass() & UC_NORMAL))
	return false;
// Forceenergy
list<BONDINST>::iterator bit, end;
end=forcelist.end();
for (bit=forcelist.begin(); bit != end; ++bit)
{
	bit->calculate(_energy);
}

// Longenergy
list<ATOM*>::iterator begin, it;
begin=atoms_placement_order.begin();
it=next_placement;
if (it == begin)
	return 0;
do
{
	--it;
	ATOM *otheratom=*it;
	assert(otheratom->get_has_coords());
	UNIT *otherunit=otheratom->get_unit();
	if ((otherunit->get_unitclass() & (UC_NORMAL | UC_PRESET)) == 0)
		continue; // UC_INVISIBLE puhul energiat ei arvutata
	if ((otherunit == unit) or (otherunit == unit->get_previousunit()))
	{ // Esimese kahe uniti sees ja vahel energiat ei arvutata.
		if (unit->get_unitclass() & UC_BEGIN)
			continue;
		if (unit->get_previousunit()->get_unitclass() & UC_BEGIN)
			continue;
	}

	if (this->has_connection_with(otheratom))
		continue;

	if (this->get_atomarg()->is_invisible_for(otheratom->get_atomarg()))
		continue;

	if (otheratom->get_atomarg()->is_invisible_for(this->get_atomarg()))
		continue;

	double dist=distance(this, otheratom);
	_energy.vdwen+=vdw_energy (this, otheratom, dist);
	_energy.couen+=coulomb_energy (this, otheratom, dist);

}
while (it != begin);
return 0;
}

bool ATOM::has_similar_forcelist(ATOM *_atom2, int _delta)
{
if (_atom2 == this)
	return false;
if (_atom2->unit == unit)
	return false;
if (_atom2->unit->get_unitdef() != unit->get_unitdef())
	return false;
list<BONDINST>::iterator it, jt;
jt=_atom2->forcelist.begin();
for (it=forcelist.begin(); it!=forcelist.end(); ++it, ++jt)
{
	if (jt==_atom2->forcelist.end())
		return false;
	if (not it->is_similar(*jt,_delta))
		return false;
}
if (jt==_atom2->forcelist.end())
	return true;
else
	return false;
}

list<BONDINST>::iterator ATOM::get_forcelist_begin()
{
return forcelist.begin();
}

list<BONDINST>::iterator ATOM::get_forcelist_end()
{
return forcelist.end();
}

int ATOM::check_all_forcelists(MOLECULE *_molecule)
{
list<ATOM*>::iterator ait;
ATOM *atoms[4];
atoms[0]=atoms[1]=atoms[2]=atoms[3]=0;
for (ait=atoms_placement_order.begin(); ait!=atoms_placement_order.end(); ++ait)
{
	ATOM *a = *ait;
	if (a->unit->get_molecule() != _molecule)
		continue;
	atoms[0]=a;
	if (! a->atomdef)
		continue;
	set<ATOM*>::iterator bit;
	for (bit=a->immediately_connected_atoms.begin();
			bit!=a->immediately_connected_atoms.end(); ++bit)
	{
		ATOM *b=*bit;
		atoms[1]=b;
		if (! b->atomdef)
			continue;
		if (b == a)
			continue;
		assert(b->atomdef);
		if (b->atoms_placement_order_id < a->atoms_placement_order_id)
			check_pair_potentials(atoms);
	}
}
return 0;
}

int ATOM::check_pair_potentials(ATOM *a[4])
{
list<BONDINST>::iterator it;
bool pair_ok=false;
for (it = a[0]->forcelist.begin(); it != a[0]->forcelist.end(); ++it)
	if (it->is_between_atoms(2, a))
		pair_ok = true;
if (not pair_ok)
{
	if(verbose_energy)
	logfile.print("Note: ATOM pair %s - %s has no bond inbetween\n",
			a[0]->get_location().c_str(),
			a[1]->get_location().c_str());
	const ATOMARG *args[4];
	for (int i=0; i<2; ++i)
		args[i]=a[i]->get_atomarg();
	args[2]=0;
	args[3]=0;
	BONDDEF *newbond=defs->find_suitable_bond(2, args);
	if (newbond)
	{
		a[0]->forcelist.push_back(
				BONDINST(_BOND_, newbond, 
					a[0], a[1], 0, 0, 0));
		if(verbose_energy)
		logfile.print("Note: Automatically added bond %s between %s-%s\n",
				newbond->get_label().c_str(),
				a[0]->get_label(),
				a[1]->get_label());
	}
	else
		if(verbose_energy)
		logfile.print("Note: No suitable automatic bond found\n");
}
set<ATOM*>::iterator cit;
for (cit=a[1]->immediately_connected_atoms.begin();
		cit!=a[1]->immediately_connected_atoms.end(); ++cit)
{
	if (!(*cit)->atomdef)
		continue;
	a[2]=*cit;
	if (a[2] == a[1])
		continue;
	if (a[2] == a[0])
		continue;
	if (a[2]->atoms_placement_order_id < a[0]->atoms_placement_order_id)
		check_triple_potentials(a);
}
return 0;
}

int ATOM::check_triple_potentials(ATOM *a[4])
{
list<BONDINST>::iterator it;
bool triple_ok=false;
for (it = a[0]->forcelist.begin(); it != a[0]->forcelist.end(); ++it)
	if (it->is_between_atoms(3, a))
		triple_ok = true;
if (not triple_ok)
{
	if(verbose_energy)
	logfile.print("Note: ATOM triple %s - %s - %s has no angle inbetween\n",
			a[0]->get_location().c_str(),
			a[1]->get_location().c_str(),
			a[2]->get_location().c_str());
	const ATOMARG *args[4];
	for (int i=0; i<3; ++i)
		args[i]=a[i]->get_atomarg();
	args[3]=0;
	BONDDEF *newbond=defs->find_suitable_bond(3, args);
	if (newbond)
	{
		a[0]->forcelist.push_back(
				BONDINST(_ANGLE_, newbond, 
					a[0], a[1], a[2], 0, 0));
		if(verbose_energy)
		logfile.print("Note: Automatically added angle %s between %s-%s-%s\n",
				newbond->get_label().c_str(),
				a[0]->get_label(),
				a[1]->get_label(),
				a[2]->get_label());
	}
	else
		if(verbose_energy)
		logfile.print("Note: No suitable automatic angle found\n");
}
set<ATOM*>::iterator dit;
for (dit=a[2]->immediately_connected_atoms.begin();
		dit!=a[2]->immediately_connected_atoms.end(); ++dit)
{
	if (! (*dit)->atomdef)
		continue;
	a[3]=*dit;
	if (a[3] == a[2])
		continue;
	if (a[3] == a[1])
		continue;
	if (a[3] == a[0])
		continue;
	if (a[3]->atoms_placement_order_id < a[0]->atoms_placement_order_id)
		check_quad_potentials(a);
}
return 0;
}

int ATOM::check_quad_potentials(ATOM *a[4])
{
list<BONDINST>::iterator it;
bool quad_ok=false;
for (it = a[0]->forcelist.begin(); it != a[0]->forcelist.end(); ++it)
	if (it->is_between_atoms(4, a))
		quad_ok = true;
if (not quad_ok)
{
	if(verbose_energy)
	logfile.print("Note: ATOM quad %s - %s - %s - %s has no dihedral inbetween\n",
			a[0]->get_location().c_str(),
			a[1]->get_location().c_str(),
			a[2]->get_location().c_str(),
			a[3]->get_location().c_str());
	const ATOMARG *args[4];
	for (int i=0; i<4; ++i)
		args[i]=a[i]->get_atomarg();
	BONDDEF *newbond=defs->find_suitable_bond(4, args);
	if (newbond)
	{
		a[0]->forcelist.push_back(
				BONDINST(_DIHEDRAL_, newbond, 
					a[0], a[1], a[2], a[3], 0));
		if(verbose_energy)
		logfile.print("Note: Automatically added dihedral %s between %s-%s-%s-%s\n",
				newbond->get_label().c_str(),
				a[0]->get_label(),
				a[1]->get_label(),
				a[2]->get_label(),
				a[3]->get_label());
	}
	else
		if(verbose_energy)
		logfile.print("Note: No suitable automatic dihedral found for %s-%s-%s-%s\n",
				a[0]->get_label(),
				a[1]->get_label(),
				a[2]->get_label(),
				a[3]->get_label());

}
return 0;
}

//========== method save_connections ==========
 
int ATOM::save_connections ( ATOM *_dep_atoms[3] )
{
	try
	{
		// Direct connections via constraints
		for (int i=0;i<3;i++ )
		{
			save_connection ( _dep_atoms[i] );
			if ( _dep_atoms[i]->atomdef )
			{
				assert ( not _dep_atoms[i]->connected_atoms.empty() );
			}
		}

		// Current atom. Guarantees that the set of connected atoms is not empty.
		save_connection ( this );
	}
	catch ( string Exception )
	{
		ostringstream msg;
		msg<<Exception;
		msg<<"\nError occurred when searching for connections of atom "
		<<unit->get_defname() <<"."<<index<<"\n";
		throw msg.str();
	}

	return 0;
}
 


